package server

import (
	"context"
	"log"
	"os"
	"strconv"
	"strings"
	"time"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"go.uber.org/zap"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	"a.yandex-team.ru/infra/porto/plugins/portostatd/pkg/iss"

	diskman_api "a.yandex-team.ru/infra/diskmanager/proto"
	porto_api "a.yandex-team.ru/infra/porto/api_go"
	nvgpu_api "a.yandex-team.ru/infra/rsm/nvgpumanager/api"
)

const (
	yasmTTL = uint(15)
)

type PortostatdServer struct {
	hostClient *internal.HostAPI
}

func InitPortostatdServer() (*PortostatdServer, error) {
	h, err := internal.CreateHostClient()
	if err != nil {
		// don't want to exit on failed hostClient init for now
		zap.S().Warnf("Restricted creation of hostClient: %v", err)
	}
	p := &PortostatdServer{
		hostClient: h,
	}
	return p, nil
}

func (s *PortostatdServer) Close() error {
	if s.hostClient != nil {
		return s.hostClient.Close()
	}
	return nil
}

// It's probably better to make this vars as part of PortostatdServer struct (add Close() method to exit properly)
var (
	portoClient   porto_api.PortoAPI
	nvgpuClient   nvgpu_api.NvGpuManagerClient
	diskmanClient diskman_api.DiskManagerClient
	ifSpeed       uint64
	cpuHasBurst   bool
)

func init() {
	// TODO: check connect timeouts and retries for both clients
	var err error
	portoClient, err = internal.CreatePortoClient()
	if err != nil {
		log.Fatalf("Failed to create porto client: %v", err)
	}
	nvgpuClient, err = internal.CreateNvgpuClient()
	if err != nil {
		log.Fatalf("Failed to create nvgpu client: %v", err)
	}
	diskmanClient, err = internal.CreateDiskmanClient()
	if err != nil {
		log.Fatalf("Failed to create diskman client: %v", err)
	}
	internal.BackboneIfname, err = backboneIfname()
	if err != nil {
		log.Printf("Failed to get backbone ifname: %v", err)
	}
	ifSpeed, err = backboneIfSpeed()
	if err != nil {
		log.Printf("Failed to get backbone iface speed: %v", err)
	}
	_, err = os.Stat("/sys/fs/cgroup/cpu/cpu.cfs_burst_usage")
	cpuHasBurst = err == nil
}

func parseUintProp(ctProps map[string]string, prop string) (uint64, error) {
	stat, err := strconv.ParseUint(ctProps[prop], 10, 64)
	if err != nil {
		return 0, status.Errorf(codes.Internal, "Can not convert to uint string value of '%v' property: %v", prop, err)
	}
	return stat, nil
}

func (s *PortostatdServer) PerformStatsCacheUpdate() error {
	start := time.Now()
	zap.S().Info("Start cache updating...")

	ctList, err := portoClient.ListContainers("*") // slots only for now
	if err != nil {
		return status.Errorf(codes.Internal, "Can not list containers: %v", err)
	}

	issPVolumes, err := iss.IssPVolumesPB()
	if err != nil {
		zap.S().Errorf("failed to get ISS PVolumes: %v", err)
	}

	issAllocations, err := formIssCtAllocations()
	if err != nil {
		zap.S().Errorf("failed to get ISS Allocations: %v", err)
	}
	zap.S().Debugf("ISS ALLOCATIONS: %#v", issAllocations)

	reqProps := []string{"state", "devices"}
	reqProps = append(reqProps, CPUStatsProps...)
	reqProps = append(reqProps, MemoryStatsProps...)
	reqProps = append(reqProps, IoStatsProps...)
	reqProps = append(reqProps, NetStatsProps...)

	ctListProps, err := portoClient.GetProperties(ctList, reqProps)
	if err != nil {
		return status.Errorf(codes.Internal, "Failed to get containers properties: %v", err)
	}

	portoVolumes, err := portoClient.ListVolumes("", "")
	if err != nil {
		zap.S().Errorf("Can not list porto volumes: %v", err)
	}

	var ctLvmStats map[string]*internal.LvmStats
	if internal.IsDiskmanRunning() {
		lvmVolumes, err := diskmanClient.ListVolumes(context.Background(), &diskman_api.ListVolumesRequest{})
		if err != nil {
			zap.S().Errorf("Can not list lvm volumes: %v", err)
		} else if len(lvmVolumes.Volumes) > 0 {
			if len(portoVolumes) > 0 {
				ctLvmStats, err = extractLvmStats(lvmVolumes.Volumes, portoVolumes)
				if err != nil {
					zap.S().Errorf("Can not extract lvm stats: %v", err)
				}
			} else {
				zap.S().Error("length of portoVolumes = 0")
			}
		}
	}

	ctCapStat, err := extractCapacityStats(issPVolumes, portoVolumes)
	if err != nil {
		zap.S().Errorf("Can not extract capacity stats: %v", err)
	}

	statsMap := make(internal.StatsMap)
	isNvgpuRunning := internal.IsNvgpuRunning()

	hostStats := internal.HostStats{}

	for ctName, ctProps := range ctListProps {
		state := ctProps["state"]
		// state is empty when container do not exist
		if state == "" || state == "stopped" || state == "starting" {
			continue // as following statistics are not available for containers in these states
		}

		var statsStorage internal.StatsStorage

		statsStorage.IOAllocStatsMap = ExtractAllocationsIO(ctName, issAllocations, diskmanClient)
		zap.S().Debugf("Allocations IO for container %s: %v", ctName, statsStorage.IOAllocStatsMap)

		statsStorage.CPUStats, err = extractCPUStats(ctProps)
		if err != nil {
			zap.S().Errorf("Can not get 'cpu_stats' of '%v' container: %v", ctName, err)
			continue
		}

		statsStorage.MemoryStats, err = extractMemoryStats(ctProps)
		if err != nil {
			zap.S().Errorf("Can not get 'memory_stats' of '%v' container: %v", ctName, err)
			continue
		}

		statsStorage.IoStats, err = extractIoStats(ctProps)
		if err != nil {
			zap.S().Errorf("Can not get 'io_stats' of '%v' container: %v", ctName, err)
			continue
		}

		statsStorage.NetStats, err = extractNetStats(ctProps, ctName)
		if err != nil {
			zap.S().Errorf("Can not get 'net_stats' of '%v' container: %v", ctName, err)
			continue
		}

		if statsStorage.NetStats.NetSnmp != nil {
			hostStats.NetTotalRetransSegs += statsStorage.NetStats.NetSnmp["RetransSegs"]
		}
		if statsStorage.NetStats.NetStat != nil {
			hostStats.NetTotalSACKReorder += statsStorage.NetStats.NetStat["TCPSACKReorder"]
		}

		if isNvgpuRunning {
			nvidiaPaths, err := extractNvidiaPaths(ctProps["devices"])
			if err == nil {
				statsStorage.GpuStats, err = getGpuStats(nvidiaPaths)
			}
			if err != nil {
				zap.S().Errorf("Can not get 'gpu_stats' of '%v' container: %v", ctName, err)
				continue
			}
		}

		statsStorage.LvmStats = ctLvmStats[ctName]
		statsStorage.CapacityStats = ctCapStat[ctName]

		// more to come

		statsMap[ctName] = &statsStorage
	}

	if s.hostClient != nil {
		err := s.hostClient.Update(&hostStats)
		if err != nil {
			zap.S().Warnf("Could not update dom0client stats: %v", err)
		} else {
			err := s.hostClient.YasmPush(yasmTTL)
			if err != nil {
				zap.S().Warnf("Could not push dom0client stats: %v", err)
			}
		}
	}

	internal.UpdateStatsCache(statsMap)

	duration := time.Since(start)
	zap.S().Infof("Cache updating took %d ms", duration.Milliseconds())

	return nil
}

func parseMemStat(input string) (map[string]uint64, error) {
	return parseStat(input, "\n", " ")
}

func parseNetStat(input string) (map[string]uint64, error) {
	return parseStat(input, ";", ":")
}

func parseStat(input, baseDelim, itemDelim string) (map[string]uint64, error) {
	res := make(map[string]uint64)

	if len(input) == 0 {
		zap.S().Debugf("Empty input string, return empty map")
		return res, nil
	}

	items := strings.Split(input, baseDelim)
	if len(items) == 1 {
		return res, status.Errorf(codes.Internal, "failed to parse input string: no '%s' symbol > %#v", baseDelim, input)
	}

	for _, item := range items {
		if len(item) == 0 {
			continue
		}
		split := strings.Split(item, itemDelim)
		if len(split) != 2 {
			return res, status.Errorf(codes.Internal, "failed to parse input string: should be 2 elements after '%s'-split > %v", itemDelim, split)
		}
		key := strings.TrimSpace(split[0])
		valueTrim := strings.TrimSpace(split[1])
		value, err := strconv.ParseUint(valueTrim, 10, 64)
		if err != nil {
			return res, status.Errorf(codes.Internal, "failed to convert to uint string value of '%v' property: %v", key, err)
		}
		res[key] = value
	}
	return res, nil
}
