package server

import (
	"testing"
)

func TestConvertNetStatStr(t *testing.T) {
	inStr := ""
	res, err := parseNetStat(inStr)
	if err != nil {
		t.Errorf("err should be nil on empty string, got: %v", err)
	}
	if len(res) != 0 {
		t.Errorf("result map should be empty on empty string, got: %v", res)
	}

	// shorter version of netstat
	inStr = "DelayedACKLocked: 197; DelayedACKLost: 0; DelayedACKs: 3555; PAWSEstab: 557; PFMemallocDrop: 0; TCPAbortOnClose: 4; TCPAbortOnData: 1; TCPAbortOnLinger: 0; TCPAbortOnMemory: 0; TCPAbortOnTimeout: 0; TCPAckCompressed: 0; TCPAutoCorking: 2050; TCPBacklogDrop: 0; TCPChallengeACK: 30; TCPDSACKIgnoredNoUndo: 0; TCPDSACKIgnoredOld: 0"
	res, err = parseNetStat(inStr)
	if err != nil {
		t.Errorf("got err: %v", err)
	}
	if len(res) == 0 {
		t.Errorf("result map is empty")
	}
	key := "TCPBacklogDrop"
	backlog, ok := res[key]
	if !ok {
		t.Errorf("%s not found in result map", key)
	}
	expect := uint64(0)
	if backlog != expect {
		t.Errorf("expected value for backlog %d, got: %d", expect, backlog)
	}

	inStr = "XXXX :197"
	_, err = parseNetStat(inStr)
	if err == nil {
		t.Errorf("err expected, got nil")
	}

	inStr = "XXXX"
	_, err = parseNetStat(inStr)
	if err == nil {
		t.Errorf("err expected, got nil")
	}

	// shorter version of /proc/net/snmp
	inStr = "InSegs: 1; OutSegs: 2; RetransSegs: 3; InErrs: 4"
	res, err = parseNetStat(inStr)
	if err != nil {
		t.Errorf("got err: %v", err)
	}
	if len(res) == 0 {
		t.Errorf("result map is empty")
	}
	key = "RetransSegs"
	retransSegs, ok := res[key]
	if !ok {
		t.Errorf("%s not found in result map", key)
	}
	expect = uint64(3)
	if retransSegs != expect {
		t.Errorf("expected value for backlog %d, got: %d", expect, backlog)
	}
}

func TestConvertMemStatStr(t *testing.T) {
	inStr := ""
	res, err := parseMemStat(inStr)
	if err != nil {
		t.Errorf("err should be nil on empty string, got: %v", err)
	}
	if len(res) != 0 {
		t.Errorf("result map should be empty on empty string, got: %v", res)
	}

	// shorter version of memory.stat
	inStr = "total_pgrefill 0\ntotal_pgsteal_kswapd 0\ntotal_pgsteal_direct 0\ntotal_pgscan_kswapd 0\n"
	res, err = parseMemStat(inStr)
	if err != nil {
		t.Errorf("got err: %v", err)
	}
	if len(res) == 0 {
		t.Errorf("result map is empty")
	}
	key := "total_pgsteal_kswapd"
	val, ok := res[key]
	if !ok {
		t.Errorf("%s not found in result map", key)
	}
	expect := uint64(0)
	if val != expect {
		t.Errorf("expected value for backlog %d, got: %d", expect, val)
	}

	inStr = "XXX 555 666"
	_, err = parseMemStat(inStr)
	if err == nil {
		t.Errorf("err expected, got nil")
	}

	inStr = "XXXX"
	_, err = parseMemStat(inStr)
	if err == nil {
		t.Errorf("err expected, got nil")
	}
}
