package internal

import (
	"fmt"
	"time"

	"a.yandex-team.ru/infra/gopcm/pkg/ipc"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal/util"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

type HostStats struct {
	NetTotalRetransSegs uint64 //Total means this metric is calculated over containers
	NetTotalSACKReorder uint64
	NetHostQdiscDrops   uint64
	CPUHostCPUWait      float64 //Host means this metric is collected from host machine itself
	MemHostNUMAStats    map[string]util.NumaBW
	//MemHostBandwith    float64
	//CPUHostIPC         ipc.Values
}

type CPUStats struct {
	Usage          uint64
	UsageSystem    uint64
	Wait           uint64
	BurstUsage     uint64
	BurstLoad      uint64
	Throttled      uint64
	GuaranteeCores float64
	LimitCores     float64
	IPC            ipc.Values // host metrics
	Instructions   uint64
	Cycles         uint64
	ThreadCount    uint64
}

func (s *CPUStats) ProtoMarshal(prev *CPUStats) *rpcpb.GetCPUStatsResponse {
	if s == nil {
		return nil
	}
	if prev == nil {
		return &rpcpb.GetCPUStatsResponse{}
	}
	dw := Delta(s.Wait, prev.Wait)
	dbu := Delta(s.BurstUsage, prev.BurstUsage)
	dbl := Delta(s.BurstLoad, prev.BurstLoad)
	dth := Delta(s.Throttled, prev.Throttled)
	dcw := Delta(dbl, dbu)
	duw := Delta(dw, dcw+dth)

	din := Delta(s.Instructions, prev.Instructions)
	dcy := Delta(s.Cycles, prev.Cycles)
	var ipc float64
	if dcy > 0 {
		ipc = float64(din) / float64(dcy)
	}

	return &rpcpb.GetCPUStatsResponse{
		CpuUsage:             Delta(s.Usage, prev.Usage),
		CpuUsageSystem:       Delta(s.UsageSystem, prev.UsageSystem),
		CpuBurstUsage:        dbu,
		CpuConstrainedWait:   dcw,
		CpuUnconstrainedWait: duw,
		CpuThrottled:         dth,
		CpuGuaranteeCores:    s.GuaranteeCores,
		CpuLimitCores:        s.LimitCores,
		ThreadCount:          s.ThreadCount,
		Ipc:                  ipc,
	}
}

type MemoryStats struct {
	Usage     uint64
	Guarantee uint64
	Limit     uint64
	Bandwith  float64 // host metric
	Stat      map[string]uint64
}

func (s *MemoryStats) ProtoMarshal(prev *MemoryStats) *rpcpb.GetMemoryStatsResponse {
	if s == nil {
		return nil
	}

	if prev == nil {
		prev = &MemoryStats{}
	}

	dKswapd, _ := handleStats(s.Stat, prev.Stat, "total_pgscan_kswapd", Delta)
	dDirect, _ := handleStats(s.Stat, prev.Stat, "total_pgscan_direct", Delta)
	dWritten, _ := handleStats(s.Stat, prev.Stat, "total_written", Delta)
	dWriteback, _ := handleStats(s.Stat, prev.Stat, "total_writeback", Delta)

	return &rpcpb.GetMemoryStatsResponse{
		MemoryUsage:             s.Usage,
		MemoryGuarantee:         s.Guarantee,
		MemoryLimit:             s.Limit,
		MemoryTotalPgscanKswapd: dKswapd,
		MemoryTotalPgscanDirect: dDirect,
		MemoryTotalWritten:      dWritten,
		MemoryTotalWriteback:    dWriteback,
	}
}

type IoStat struct {
	Dev string
	Val uint64
}

func (s *IoStat) ProtoMarshal() *rpcpb.GetIoStatResponse_IoStatRsp {
	if s == nil {
		return nil
	}
	return &rpcpb.GetIoStatResponse_IoStatRsp{
		Dev: s.Dev,
		Val: s.Val,
	}
}

type IoStats struct {
	Read  []IoStat
	Write []IoStat
	Ops   []IoStat
	Time  []IoStat
}

func (s *IoStats) protoMarshal(ioStats []IoStat) *rpcpb.GetIoStatResponse {
	protoIoStats := make([]*rpcpb.GetIoStatResponse_IoStatRsp, 0, len(ioStats))
	for _, v := range ioStats {
		protoIoStats = append(protoIoStats, v.ProtoMarshal())
	}
	return &rpcpb.GetIoStatResponse{IoStat: protoIoStats}

}

func (s *IoStats) ProtoMarshal() *rpcpb.GetIoStatsResponse {
	if s == nil {
		return nil
	}
	return &rpcpb.GetIoStatsResponse{
		IoRead:  s.protoMarshal(s.Read),
		IoWrite: s.protoMarshal(s.Write),
		IoOps:   s.protoMarshal(s.Ops),
		IoTime:  s.protoMarshal(s.Time),
	}
}

type NetInterfaceStats struct {
	RxBytes   uint64
	TxBytes   uint64
	RxPackets uint64
	TxPackets uint64
	RxDrops   uint64
	TxDrops   uint64
}

type NetStats struct {
	IfaceStat          map[string]*NetInterfaceStats
	UplinkTxBytes      uint64
	UplinkRxBytes      uint64
	UplinkTxLimit      uint64
	UplinkRxLimit      uint64
	UplinkTxOverlimits *uint64
	UplinkRxOverlimits *uint64
	NetStat            map[string]uint64
	NetSnmp            map[string]uint64
	TxSpeedHgram       *rpcpb.GetNetStatHgramResponse
	RxSpeedHgram       *rpcpb.GetNetStatHgramResponse
}

func (s *NetStats) utilization(bytes, limit uint64, d time.Duration) uint64 {
	if limit == 0 || d.Seconds() <= 0 {
		return 0
	}
	speed := float64(bytes) / d.Seconds()
	return uint64(100 * (speed / float64(limit)))
}

func processInterfaceStats(ifaceName string, curr *NetInterfaceStats, prev *NetInterfaceStats) *rpcpb.GetInterfaceStat_InterfaceStat {
	ret := &rpcpb.GetInterfaceStat_InterfaceStat{}
	properties := []struct {
		curr uint64
		prev uint64
		ret  *uint64
	}{
		{curr.RxBytes, prev.RxBytes, &ret.RxBytes},
		{curr.TxBytes, prev.TxBytes, &ret.TxBytes},
		{curr.RxPackets, prev.RxPackets, &ret.RxPackets},
		{curr.TxPackets, prev.TxPackets, &ret.TxPackets},
		{curr.RxDrops, prev.RxDrops, &ret.RxDrops},
		{curr.TxDrops, prev.TxDrops, &ret.TxDrops},
	}
	for _, p := range properties {
		if p.curr >= p.prev {
			*p.ret = p.curr - p.prev
		} else {
			return nil
		}
	}
	ret.Ifname = ifaceName
	return ret
}

//This function in now only used in net_stats
//Expand it to whole package if refactoring occurs
func handleStats(now map[string]uint64, then map[string]uint64, field string, handlefunc func(uint64, uint64) uint64) (uint64, error) {
	c, ok := now[field]
	if !ok {
		zap.S().Debugf("%s do not exist, use default", field)
	}
	p, ok := then[field]
	if !ok {
		return 0, fmt.Errorf("skipped %s", field)
	}
	return handlefunc(c, p), nil
}

//This function in now only used in net_stats
//Expand it to whole package if refactoring occurs
func Delta(c uint64, p uint64) uint64 {
	if c > p {
		return c - p
	}
	return 0
}

//This function in now only used in net_stats
//Expand it to whole package if refactoring occurs
func Current(c uint64, p uint64) uint64 {
	return c
}

func (s *NetStats) ProtoMarshal(prev *NetStats, t time.Time) (resp *rpcpb.GetNetStatsResponse) {
	if s == nil {
		return nil
	}
	if prev == nil {
		prev = &NetStats{}
	}
	d := time.Since(t)

	resp = &rpcpb.GetNetStatsResponse{}

	resp.NetUplinkRxBytes = &rpcpb.GetNetStatResponse{Val: Delta(s.UplinkRxBytes, prev.UplinkRxBytes)}
	resp.NetUplinkTxBytes = &rpcpb.GetNetStatResponse{Val: Delta(s.UplinkTxBytes, prev.UplinkTxBytes)}
	resp.NetUplinkRxUtilization = &rpcpb.GetNetStatResponse{Val: s.utilization(resp.NetUplinkRxBytes.Val, s.UplinkRxLimit, d)}
	resp.NetUplinkTxUtilization = &rpcpb.GetNetStatResponse{Val: s.utilization(resp.NetUplinkTxBytes.Val, s.UplinkTxLimit, d)}

	if s.UplinkRxOverlimits != nil && prev.UplinkRxOverlimits != nil {
		resp.NetUplinkRxOverlimits = &rpcpb.GetNetStatResponse{Val: Delta(*s.UplinkRxOverlimits, *prev.UplinkRxOverlimits)}
	}

	if s.UplinkTxOverlimits != nil && prev.UplinkTxOverlimits != nil {
		resp.NetUplinkTxOverlimits = &rpcpb.GetNetStatResponse{Val: Delta(*s.UplinkTxOverlimits, *prev.UplinkTxOverlimits)}
	}

	v, err := handleStats(s.NetStat, prev.NetStat, "TCPBacklogDrop", Delta)
	if err != nil {
		resp.NetTcpBacklogDrop = nil
	} else {
		resp.NetTcpBacklogDrop = &rpcpb.GetNetStatResponse{Val: v}
	}

	v, err = handleStats(s.NetStat, prev.NetStat, "TCPSynRetrans", Delta)
	if err != nil {
		resp.NetTcpSynRetrans = nil
	} else {
		resp.NetTcpSynRetrans = &rpcpb.GetNetStatResponse{Val: v}
	}

	v, err = handleStats(s.NetStat, prev.NetStat, "TCPSACKReorder", Delta)
	if err != nil {
		resp.NetTcpSackReorder = nil
	} else {
		resp.NetTcpSackReorder = &rpcpb.GetNetStatResponse{Val: v}
	}

	v, err = handleStats(s.NetSnmp, prev.NetSnmp, "RetransSegs", Delta)
	if err != nil {
		resp.NetRetransSegs = nil
	} else {
		resp.NetRetransSegs = &rpcpb.GetNetStatResponse{Val: v}
	}

	v, err = handleStats(s.NetSnmp, prev.NetSnmp, "CurrEstab", Current)
	if err != nil {
		resp.NetCurrEstab = nil
	} else {
		resp.NetCurrEstab = &rpcpb.GetNetStatResponse{Val: v}
	}

	netInterfaces := make([]*rpcpb.GetInterfaceStat_InterfaceStat, 0, len(s.IfaceStat))

	for interfaceName, ifaceStats := range s.IfaceStat {
		prevIfaceStat := prev.IfaceStat[interfaceName]
		if prevIfaceStat != nil {
			ifaceInfo := processInterfaceStats(interfaceName, ifaceStats, prevIfaceStat)
			if ifaceInfo != nil {
				netInterfaces = append(netInterfaces, ifaceInfo)
			}
		}
	}

	resp.NetInterfaceStats = &rpcpb.GetInterfaceStat{NetInterfaces: netInterfaces}
	resp.NetRxSpeedHgram = s.RxSpeedHgram
	resp.NetTxSpeedHgram = s.TxSpeedHgram

	return
}

type GpuStat struct {
	Dev string
	Val uint64
}

func (s *GpuStat) ProtoMarshal() *rpcpb.GetGpuStatResponse_GpuStatRsp {
	if s == nil {
		return nil
	}
	return &rpcpb.GetGpuStatResponse_GpuStatRsp{
		Dev: s.Dev,
		Val: s.Val,
	}
}

type GpuStats struct {
	Temperature       []GpuStat
	Utilization       []GpuStat
	MemoryUtilization []GpuStat
}

func (s *GpuStats) protoMarshalGpuStat(gpuStats []GpuStat) *rpcpb.GetGpuStatResponse {
	protoGpuStats := make([]*rpcpb.GetGpuStatResponse_GpuStatRsp, 0, len(gpuStats))
	for _, v := range gpuStats {
		protoGpuStats = append(protoGpuStats, v.ProtoMarshal())
	}
	return &rpcpb.GetGpuStatResponse{GpuStat: protoGpuStats}
}

func (s *GpuStats) ProtoMarshal() *rpcpb.GetGpuStatsResponse {
	if s == nil {
		return nil
	}
	return &rpcpb.GetGpuStatsResponse{
		GpuTemperature:       s.protoMarshalGpuStat(s.Temperature),
		GpuUtilization:       s.protoMarshalGpuStat(s.Utilization),
		GpuMemoryUtilization: s.protoMarshalGpuStat(s.MemoryUtilization),
	}
}

type LvmVolumeStats struct {
	VolName        string
	AvailableBytes uint64
	UsedBytes      uint64
	ReadBytes      uint64
	WriteBytes     uint64
	ReadOps        uint64
	WriteOps       uint64
	IoInProgress   uint64
	Await          uint64
}

func (s *LvmVolumeStats) ProtoMarshal() *rpcpb.GetLvmStatsResponse_VolumeStats {
	if s == nil {
		return nil
	}
	return &rpcpb.GetLvmStatsResponse_VolumeStats{
		VolName:        s.VolName,
		AvailableBytes: s.AvailableBytes,
		UsedBytes:      s.UsedBytes,
		ReadBytes:      s.ReadBytes,
		WriteBytes:     s.WriteBytes,
		ReadOps:        s.ReadOps,
		WriteOps:       s.WriteOps,
		IoInProgress:   s.IoInProgress,
		Await:          s.Await,
	}
}

type LvmStats struct {
	VolStats []LvmVolumeStats
}

func (s *LvmStats) ProtoMarshal() *rpcpb.GetLvmStatsResponse {
	if s == nil {
		return nil
	}
	volStats := make([]*rpcpb.GetLvmStatsResponse_VolumeStats, len(s.VolStats))
	for i := range s.VolStats {
		volStats[i] = s.VolStats[i].ProtoMarshal()
	}
	return &rpcpb.GetLvmStatsResponse{
		VolStats: volStats,
	}
}

type CapStats struct {
	VolCapStats []*VolCapStat
}

type VolCapStat struct {
	VolName   string
	PercUsage uint64
}

func (s *VolCapStat) ProtoMarshal() *rpcpb.GetCapacityStatsResponse_CapStat {
	if s == nil {
		return nil
	}
	return &rpcpb.GetCapacityStatsResponse_CapStat{
		VolName:   s.VolName,
		PercUsage: s.PercUsage,
	}
}

func (s *CapStats) ProtoMarshal() *rpcpb.GetCapacityStatsResponse {
	if s == nil {
		return nil
	}

	volStats := make([]*rpcpb.GetCapacityStatsResponse_CapStat, len(s.VolCapStats))
	for i := range s.VolCapStats {
		volStats[i] = s.VolCapStats[i].ProtoMarshal()
	}
	return &rpcpb.GetCapacityStatsResponse{
		CapacityStats: volStats,
	}
}

// key is AllocName
type IOAllocStatsMap map[string]*IOAllocStats

type IOAllocStats struct {
	AllocName     string
	ReadBwLimit   uint64
	ReadBytes     uint64
	WriteBwLimit  uint64
	WriteBytes    uint64
	ReadOpsLimit  uint64
	WriteOpsLimit uint64
	ReadOps       uint64
	WriteOps      uint64
}

func (ioas *IOAllocStats) Add(stats *IOAllocStats) {
	ioas.ReadBwLimit += stats.ReadBwLimit
	ioas.ReadBytes += stats.ReadBytes
	ioas.WriteBwLimit += stats.WriteBwLimit
	ioas.WriteBytes += stats.WriteBytes
	ioas.ReadOpsLimit += stats.ReadOpsLimit
	ioas.WriteOpsLimit += stats.WriteOpsLimit
	ioas.ReadOps += stats.ReadOps
	ioas.WriteOps += stats.WriteOps
}

func SubIOAlloc(decreasing, deductible *IOAllocStats) *IOAllocStats {
	r := &IOAllocStats{}
	r.AllocName = decreasing.AllocName

	r.ReadBwLimit = decreasing.ReadBwLimit
	r.ReadOpsLimit = decreasing.ReadOpsLimit
	r.WriteBwLimit = decreasing.WriteBwLimit
	r.WriteOpsLimit = decreasing.WriteOpsLimit

	r.ReadBytes = decreasing.ReadBytes - deductible.ReadBytes
	r.WriteBytes = decreasing.WriteBytes - deductible.WriteBytes
	r.ReadOps = decreasing.ReadOps - deductible.ReadOps
	r.WriteOps = decreasing.WriteOps - deductible.WriteOps
	return r
}

func bw(val uint64, dt time.Duration) uint64 {
	if dt.Seconds() <= 0 {
		return 0
	}
	return val / uint64(dt.Seconds())

}

func (ioasm IOAllocStatsMap) ProtoMarshal(prev IOAllocStatsMap, t time.Time) *rpcpb.GetIOAllocStatsResponse {
	if prev == nil {
		return nil
	}
	res := &rpcpb.GetIOAllocStatsResponse{}
	res.IoAllocStats = []*rpcpb.IOAllocStat{}

	dt := time.Since(t)

	for storage, values := range ioasm {
		if prevVal, ok := prev[storage]; ok {
			r := SubIOAlloc(values, prevVal)
			rp := &rpcpb.IOAllocStat{
				AllocName:     r.AllocName,
				ReadBwLimit:   r.ReadBwLimit,
				WriteBwLimit:  r.WriteBwLimit,
				ReadOpsLimit:  r.ReadOpsLimit,
				WriteOpsLimit: r.WriteOpsLimit,
				ReadBwBytes:   bw(r.ReadBytes, dt),
				WriteBwBytes:  bw(r.WriteBytes, dt),
				ReadOps:       bw(r.ReadOps, dt),
				WriteOps:      bw(r.WriteOps, dt),
			}
			res.IoAllocStats = append(res.IoAllocStats, rp)
		} else {
			continue
		}
	}
	return res
}
