package internal

import (
	"sync"
	"time"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

type StatsProtoStorage struct {
	CPUStats      *rpcpb.GetCPUStatsResponse
	MemoryStats   *rpcpb.GetMemoryStatsResponse
	IoStats       *rpcpb.GetIoStatsResponse
	NetStats      *rpcpb.GetNetStatsResponse
	GpuStats      *rpcpb.GetGpuStatsResponse
	LvmStats      *rpcpb.GetLvmStatsResponse
	CapacityStats *rpcpb.GetCapacityStatsResponse
	IOAllocStats  *rpcpb.GetIOAllocStatsResponse
	// more to come
}

type StatsStorage struct {
	CPUStats        *CPUStats
	MemoryStats     *MemoryStats
	IoStats         *IoStats
	NetStats        *NetStats
	GpuStats        *GpuStats
	LvmStats        *LvmStats
	CapacityStats   *CapStats
	IOAllocStatsMap IOAllocStatsMap
	// more to come
}

type HostStatsStorage struct {
	HostStats *HostStats
}

func (s *StatsStorage) ProtoMarshal(prev *StatsStorage, t time.Time) *StatsProtoStorage {
	return &StatsProtoStorage{
		CPUStats:      s.CPUStats.ProtoMarshal(prev.CPUStats),
		MemoryStats:   s.MemoryStats.ProtoMarshal(prev.MemoryStats),
		IoStats:       s.IoStats.ProtoMarshal(),
		NetStats:      s.NetStats.ProtoMarshal(prev.NetStats, t),
		GpuStats:      s.GpuStats.ProtoMarshal(),
		LvmStats:      s.LvmStats.ProtoMarshal(),
		CapacityStats: s.CapacityStats.ProtoMarshal(),
		IOAllocStats:  s.IOAllocStatsMap.ProtoMarshal(prev.IOAllocStatsMap, t),
	}
}

// container name to it's stats
type StatsMap map[string]*StatsStorage

type statsProtoMap map[string]*StatsProtoStorage

func (s StatsMap) ProtoMarshal(prev StatsMap, t time.Time) statsProtoMap {
	stats := make(statsProtoMap, len(s))
	for k, v := range s {
		prevV := prev[k]
		if prevV == nil {
			prevV = &StatsStorage{}
		}
		stats[k] = v.ProtoMarshal(prevV, t)
	}
	return stats
}

type ContainersList []string

func (cl ContainersList) ProtoMarshal() *rpcpb.ContainersList {
	return &rpcpb.ContainersList{CtNames: cl}
}

type statsCache struct {
	sync.RWMutex
	storage      StatsMap
	storageProto statsProtoMap
	storageTime  time.Time
}

var cache statsCache

func UpdateStatsCache(statsMap StatsMap) {
	cache.Lock()
	defer cache.Unlock()

	cache.storageProto = statsMap.ProtoMarshal(cache.storage, cache.storageTime)
	cache.storage = statsMap
	cache.storageTime = time.Now()
}

func GetContainerNames() (*rpcpb.ContainersList, error) {
	cache.RLock()
	storageProto := cache.storageProto
	cache.RUnlock()

	list := make([]string, 0, len(storageProto))
	for ctName := range storageProto {
		list = append(list, ctName)
	}

	return &rpcpb.ContainersList{CtNames: list}, nil
}

func GetStatsStorage(ctName string) (*StatsProtoStorage, bool) {
	cache.RLock()
	storageProto := cache.storageProto
	cache.RUnlock()

	val, ok := storageProto[ctName]

	return val, ok
}

func GetExistentStatsStorage(ctName string) (*StatsProtoStorage, error) {
	statsStorage, ok := GetStatsStorage(ctName)
	if !ok {
		return nil, status.Errorf(codes.Internal, "Can not find container with name %v", ctName)
	}
	return statsStorage, nil
}

func GetCPUStatsStorage(ctName string) (*rpcpb.GetCPUStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.CPUStats == nil {
		return nil, status.Errorf(codes.Internal, "CPUStats is not initialized for %v", ctName)
	}

	return statsStorage.CPUStats, nil
}

func GetMemoryStatsStorage(ctName string) (*rpcpb.GetMemoryStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.MemoryStats == nil {
		return nil, status.Errorf(codes.Internal, "MemoryStats is not initialized for %v", ctName)
	}

	return statsStorage.MemoryStats, nil
}

func GetIoStatsStorage(ctName string) (*rpcpb.GetIoStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.IoStats == nil {
		return nil, status.Errorf(codes.Internal, "IoStats is not initialized for %v", ctName)
	}

	return statsStorage.IoStats, nil
}

func GetNetStatsStorage(ctName string) (*rpcpb.GetNetStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.NetStats == nil {
		return nil, status.Errorf(codes.Internal, "NetStats is not initialized for %v", ctName)
	}

	return statsStorage.NetStats, nil
}

func GetGpuStatsStorage(ctName string) (*rpcpb.GetGpuStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.GpuStats == nil {
		return nil, status.Errorf(codes.Internal, "GpuStats is not initialized for %v", ctName)
	}

	return statsStorage.GpuStats, nil
}

func GetLvmStatsStorage(ctName string) (*rpcpb.GetLvmStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.LvmStats == nil {
		return nil, status.Errorf(codes.Internal, "LvmStats is not initialized for %v", ctName)
	}

	return statsStorage.LvmStats, nil
}

func GetCapacityStatsStorage(ctName string) (*rpcpb.GetCapacityStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.CapacityStats == nil {
		return nil, status.Errorf(codes.Internal, "capacity stats is not initialized for %v", ctName)
	}
	return statsStorage.CapacityStats, nil
}

func GetIOAllocStatsStorage(ctName string) (*rpcpb.GetIOAllocStatsResponse, error) {
	statsStorage, err := GetExistentStatsStorage(ctName)
	if err != nil {
		return nil, err
	}
	if statsStorage.IOAllocStats == nil {
		return nil, status.Errorf(codes.Internal, "LvmStats is not initialized for %v", ctName)
	}
	return statsStorage.IOAllocStats, nil
}
