package util

import (
	"errors"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"
	"strconv"
	"strings"
	"syscall"

	"go.uber.org/zap"
)

var (
	resctrlpath     = "/sys/fs/resctrl"
	ErrResctrlEmpty = errors.New("resctrl contains error")
)

type L3Bytes struct {
	Local *uint64
	Total *uint64
}

type NumaBW L3Bytes

func GetCPUWait() (float64, error) {
	CoresPerNsec := 1e+9
	fpath := "/sys/fs/cgroup"
	dirs := []string{"cpuacct", "cpu", "cpu,cpuacct"}
	for _, f := range dirs {
		if _, err := os.Stat(filepath.Join(fpath, f, "cpuacct.wait")); !errors.Is(err, os.ErrNotExist) {
			strnum, err := ioutil.ReadFile(filepath.Join(fpath, f, "cpuacct.wait"))
			if err != nil {
				return 0, err
			}
			cpuWaitTime, err := strconv.ParseInt(strings.TrimSpace(string(strnum)), 10, 64)
			if err != nil {
				return 0, err
			}
			return float64(cpuWaitTime) / CoresPerNsec, nil
		}
	}
	return 0, errors.New("no files available to extract CpuWait")
}

func GetIntStatFile(filepath string) (uint64, error) {
	strnum, err := ioutil.ReadFile(path.Clean(filepath))
	if err != nil {
		return 0, err
	}
	val, err := strconv.ParseInt(strings.TrimSpace(string(strnum)), 10, 64)
	if err != nil {
		if strings.TrimSpace(string(strnum)) == "Error" {
			return 0, ErrResctrlEmpty
		}
		return 0, err
	}
	return uint64(val), nil
}

func MountResctrl() (err error) {
	// try mounting resctrl interface
	err = syscall.Mount("resctrl", resctrlpath, "resctrl", 0, "")
	if err != nil {
		if errors.Is(err, syscall.EBUSY) {
			err = nil
			zap.S().Info("resctrl already mounted, skipped")
		}
	}
	return
}

func GetNUMAStats() (numabw map[string]NumaBW, err error) {
	// GetNUMAStats (mbml, mbmr, error)
	l3Files, err := ioutil.ReadDir(path.Join(resctrlpath, "mon_data"))
	if err != nil {
		return nil, err
	}

	l3s := map[int]L3Bytes{}

	for _, f := range l3Files {
		if f.IsDir() {
			l3 := L3Bytes{}
			l3Path := path.Join(resctrlpath, "mon_data", f.Name(), "mbm_local_bytes")
			mbml, err := GetIntStatFile(l3Path)
			if err != nil {
				if !errors.Is(err, ErrResctrlEmpty) {
					return nil, err
				}
				zap.S().Warnf("resctrl shows Error at %s", l3Path)
				l3.Local = nil
			} else {
				l3.Local = &mbml
			}

			l3Path = path.Join(resctrlpath, "mon_data", f.Name(), "mbm_total_bytes")
			mbmt, err := GetIntStatFile(l3Path)
			if err != nil {
				if !errors.Is(err, ErrResctrlEmpty) {
					return nil, err
				}
				zap.S().Warnf("resctrl shows Error at %s", l3Path)
				l3.Total = nil
			} else {
				l3.Total = &mbmt
			}

			l3id, err := strconv.Atoi(strings.TrimPrefix(f.Name(), "mon_L3_"))
			if err != nil {
				return nil, err
			}
			l3s[l3id] = l3
		}
	}

	numabw, err = NUMAbwFromL3(l3s)
	if err != nil {
		return nil, err
	}

	return
}

func NUMAbwFromL3(l3s map[int]L3Bytes) (numabw map[string]NumaBW, err error) {
	numaFiles, err := ioutil.ReadDir(path.Clean("/sys/bus/node/devices"))
	if err != nil {
		return nil, err
	}

	numabw = map[string]NumaBW{}

	for _, numa := range numaFiles {
		l3allocids := make(map[int]struct{})
		cpuFiles, err := filepath.Glob(path.Join("/sys/bus/node/devices", numa.Name(), "cpu*/cache/index3/id"))
		if err != nil {
			return nil, err
		}
		for _, cpu := range cpuFiles {
			l3id, err := GetIntStatFile(cpu)
			if err != nil {
				return nil, err
			}
			l3allocids[int(l3id)] = struct{}{}
		}

		a, b := uint64(0), uint64(0)
		t := NumaBW{&a, &b}

		for l3id := range l3allocids {
			if t.Local != nil {
				if l3s[l3id].Local == nil {
					t.Local = nil
				} else {
					*t.Local += *l3s[l3id].Local
				}
			}
			if t.Total != nil {
				if l3s[l3id].Total == nil {
					t.Total = nil
				} else {
					*t.Total += *l3s[l3id].Total
				}
			}
		}
		numabw[numa.Name()] = t
	}
	return
}
