package util

import (
	"bytes"
	"encoding/binary"

	"github.com/vishvananda/netlink"
	"github.com/vishvananda/netlink/nl"
	"go.uber.org/zap"
	"golang.org/x/sys/unix"
)

func GetQdiscDrops() uint64 {
	type tcStats struct {
		Bytes      uint64 // Number of enqueued bytes
		Packets    uint32 // Number of enqueued packets
		Drops      uint32 // Packets dropped because of lack of resources
		Overlimits uint32 // Number of throttle events when this flow goes out of allocated bandwidth
		Bps        uint32 // Current flow byte rate
		Pps        uint32 // Current flow packet rate
		Qlen       uint32
		Backlog    uint32
	}

	// This section of code can be used if you need drops from a particular network interface
	// ensureIndex := func(link *netlink.LinkAttrs) {
	// 	if link != nil && link.Index == 0 {
	// 		newlink, _ := netlink.LinkByName(link.Name)
	// 		if newlink != nil {
	// 			link.Index = newlink.Attrs().Index
	// 		}
	// 	}
	// }

	// link, err := netlink.LinkByName("")
	// if err != nil {
	// 	zap.S().Warnf("Error getting link: %v", err)
	// }
	req := nl.NewNetlinkRequest(unix.RTM_GETQDISC, unix.NLM_F_DUMP)
	msg := &nl.TcMsg{
		Family: nl.FAMILY_ALL,
		Parent: 1,
	}

	// if link != nil {
	// 	base := link.Attrs()
	// 	ensureIndex(base)
	// 	msg.Ifindex = int32(base.Index)
	// }
	req.AddData(msg)

	msgs, err := req.Execute(unix.NETLINK_ROUTE, unix.RTM_NEWQDISC)
	if err != nil {
		zap.S().Error(err)
	}

	var drops uint64
	for _, m := range msgs {
		msg := nl.DeserializeTcMsg(m)
		if msg.Parent != netlink.HANDLE_ROOT {
			continue
		}

		attrs, err := nl.ParseRouteAttr(m[msg.Len():])
		if err != nil {
			zap.S().Error(err)
		}
		for _, attr := range attrs {
			if attr.Attr.Type == nl.TCA_STATS {
				buf := &bytes.Buffer{}
				buf.Write(attr.Value)
				native := nl.NativeEndian()
				tcStats := &tcStats{}
				if err := binary.Read(buf, native, tcStats); err != nil {
					zap.S().Errorf("Could not parse tcStats: %v", err)
				} else {
					drops += uint64(tcStats.Drops)
				}
			}
		}
	}

	return drops
}
