package diskstat

import (
	"bufio"
	"fmt"
	"io"
	"os"
)

const DiskStatPath = "/proc/diskstats"

type Device struct {
	Maj uint32
	Min uint32
}

type DiskStatMap map[Device]*DiskStatRecord

type DiskStatRecord struct {
	// times are in ms
	MajNum           uint32
	MinNum           uint32
	Name             string
	ReadSuccess      uint64
	ReadMerged       uint64
	ReadSectors      uint64
	TimeInRead       uint64
	WriteSuccess     uint64
	WriteMerged      uint64
	WriteSectors     uint64
	TimeInWrite      uint64
	IoInProgress     uint64
	TimeInIO         uint64
	WeightedTimeInIO uint64
	DiscardSuccess   uint64
	DiscardMerged    uint64
	DiscardSectors   uint64
	TimeInDiscard    uint64
	FlushSuccess     uint64
	TimeInFlush      uint64
}

func FromReader(r io.Reader) (DiskStatMap, error) {
	d := make(DiskStatMap)
	scanner := bufio.NewScanner(r)
	for scanner.Scan() {
		record := &DiskStatRecord{}
		_, err := fmt.Sscan(
			scanner.Text(),
			&record.MajNum,
			&record.MinNum,
			&record.Name,
			&record.ReadSuccess,
			&record.ReadMerged,
			&record.ReadSectors,
			&record.TimeInRead,
			&record.WriteSuccess,
			&record.WriteMerged,
			&record.WriteSectors,
			&record.TimeInWrite,
			&record.IoInProgress,
			&record.TimeInIO,
			&record.WeightedTimeInIO,
			&record.DiscardSuccess,
			&record.DiscardMerged,
			&record.DiscardSectors,
			&record.TimeInDiscard,
			&record.FlushSuccess,
			&record.TimeInFlush,
		)
		if err != nil {
			return d, err
		}

		key := Device{
			Maj: record.MajNum,
			Min: record.MinNum,
		}
		d[key] = record
	}
	return d, nil
}

func FromFile(path string) (DiskStatMap, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, err
	}
	defer f.Close()
	return FromReader(f)
}

func ReadSystem() (DiskStatMap, error) {
	return FromFile(DiskStatPath)
}
