#!/usr/bin/env bash

# rewrite https://github.com/moby/moby/blob/master/contrib/download-frozen-image-v2.sh
set -eo pipefail

# check if essential commands are in our PATH
for cmd in curl jq; do
	if ! command -v $cmd &> /dev/null; then
		echo >&2 "error: \"$cmd\" not found!"
		exit 1
	fi
done

usage() {
	echo "usage: $0 [-run] [-rtc|-custom] resultLayer.tar image[:tag][@digest]"
	echo "       $0 layer.tar ubuntu:latest@sha256:703218c0465075f4425e58fac086e09e1de5c340b12976ab9eb8ad26615c3715"
    echo '    -rtc needs env var $DOCKER_TOKEN'
    echo '    https://wiki.yandex-team.ru/docker-registry/'
    echo '    -custom needs env vars $REGISTRY_BASE, $AUTH_HEADER(example: "Authorization: OAuth"), and $DOCKER_TOKEN or $AUTH_BASE and $AUTH_SERVICE'
	exit 1
}

if [ "$1" == "-run" ]; then
    run="1"
    shift
fi

if [ "$1" == "-rtc" ]; then
    shift
    if [ -z "$DOCKER_TOKEN" ]; then
        usage
    fi
    REGISTRY_BASE='https://registry.yandex.net'
    AUTH_HEADER="Authorization: OAuth"
elif [ "$1" == "-custom" ]; then
    shift
    if [ -z "$REGISTRY_BASE" ] || [ -z "$AUTH_HEADER" ]; then
        usage
    fi

    if [ -z "$DOCKER_TOKEN" ]; then
        if [ -z "$AUTH_BASE" ] || [ -z "$AUTH_SERVICE" ]; then
            usage
        fi
    fi
else
    DOCKER_TOKEN=''
    REGISTRY_BASE='https://registry-1.docker.io'
    AUTH_BASE='https://auth.docker.io'
    AUTH_SERVICE='registry.docker.io'
    AUTH_HEADER="Authorization: Bearer"
fi

if [ -z "$1" ]; then
    usage
fi

result_layer=$1
shift

if [ -z "$1" ]; then
    usage
fi

dir=$(mktemp -dt docker_to_porto_XXXX)

# https://github.com/moby/moby/issues/33700
fetch_blob() {
    if [ -z "$DOCKER_TOKEN" ]; then
	local DOCKER_TOKEN="$1"; fi
	shift
	local image="$1"
	shift
	local digest="$1"
	shift
	local targetFile="$1"
	shift
	local curlArgs=("$@")

	local curlHeaders
	curlHeaders="$(
		curl -S "${curlArgs[@]}" \
			-H "$AUTH_HEADER $DOCKER_TOKEN" \
			"$REGISTRY_BASE/v2/$image/blobs/$digest" \
			-o "$targetFile" \
			-D-
	)"
	curlHeaders="$(echo "$curlHeaders" | tr -d '\r')"
	if grep -qE "^HTTP/[0-9].[0-9] 3" <<< "$curlHeaders"; then
		rm -f "$targetFile"

		local blobRedirect
		blobRedirect="$(echo "$curlHeaders" | awk -F ': ' 'tolower($1) == "location" { print $2; exit }')"
		if [ -z "$blobRedirect" ]; then
			echo >&2 "error: failed fetching '$image' blob '$digest'"
			echo "$curlHeaders" | head -1 >&2
			return 1
		fi

		curl -fSL "${curlArgs[@]}" \
			"$blobRedirect" \
			-o "$targetFile"
	fi
}

# handle 'application/vnd.docker.distribution.manifest.v2+json' manifest
handle_single_manifest_v2() {
	local manifestJson="$1"
	shift

	local configDigest
	configDigest="$(echo "$manifestJson" | jq --raw-output '.config.digest')"
	local imageId="${configDigest#*:}" # strip off "sha256:"

	local configFile="$imageId.json"
	fetch_blob "$DOCKER_TOKEN" "$image" "$configDigest" "$dir/$configFile" -s

	local layersFs
	layersFs="$(echo "$manifestJson" | jq --raw-output --compact-output '.layers[]')"
	local IFS="$newlineIFS"
	local layers
	mapfile -t layers <<< "$layersFs"
	unset IFS

	echo "Downloading '$imageIdentifier' (${#layers[@]} layers)..."
	local layerId=
	local layerFiles=()
	for i in "${!layers[@]}"; do
        echo $(($i+1))/"${#layers[@]}"
		local layerMeta="${layers[$i]}"

		local layerMediaType
		layerMediaType="$(echo "$layerMeta" | jq --raw-output '.mediaType')"
		local layerDigest
		layerDigest="$(echo "$layerMeta" | jq --raw-output '.digest')"

		# save the previous layer's ID
		local parentId="$layerId"
		# create a new fake layer ID based on this layer's digest and the previous layer's fake ID
		layerId="$(echo "$parentId"$'\n'"$layerDigest" | sha256sum | cut -d' ' -f1)"
		# this accounts for the possibility that an image contains the same layer twice (and thus has a duplicate digest value)

		mkdir -p "$dir/$layerId"

		case "$layerMediaType" in
			application/vnd.docker.image.rootfs.diff.tar.gzip)
				local layerTar="$layerId/layer.tar"
				layerFiles=("${layerFiles[@]}" "$layerTar")
				# TODO figure out why "-C -" doesn't work here
				# "curl: (33) HTTP server doesn't seem to support byte ranges. Cannot resume."
				# "HTTP/1.1 416 Requested Range Not Satisfiable"
				if [ -f "$dir/$layerTar" ]; then
					# TODO hackpatch for no -C support :'(
					echo "skipping existing ${layerId:0:12}"
					continue
				fi
                if [ -z "$DOCKER_TOKEN" ]; then
                    local DOCKER_TOKEN 
                    DOCKER_TOKEN="$(curl -fsSL "$AUTH_BASE/token?service=$AUTH_SERVICE&scope=repository:$image:pull" | jq --raw-output '.token')"
                fi
				fetch_blob "$DOCKER_TOKEN" "$image" "$layerDigest" "$dir/$layerTar" --progress-bar
				;;

			*)
				echo >&2 "error: unknown layer mediaType ($imageIdentifier, $layerDigest): '$layerMediaType'"
                rm -rf $dir
				exit 1
				;;
		esac
	done

    for layerFile in ${layerFiles[@]}; do layers_paths="$layers_paths -L $dir/$layerFile"; done
}

get_target_arch() {
	if [ -n "${TARGETARCH:-}" ]; then
		echo "${TARGETARCH}"
		return 0
	fi

	if type go > /dev/null; then
		go env GOARCH
		return 0
	fi

	if type dpkg > /dev/null; then
		debArch="$(dpkg --print-architecture)"
		case "${debArch}" in
			armel | armhf)
				echo "arm"
				return 0
				;;
			*64el)
				echo "${debArch%el}le"
				return 0
				;;
			*)
				echo "${debArch}"
				return 0
				;;
		esac
	fi

	if type uname > /dev/null; then
		uArch="$(uname -m)"
		case "${uArch}" in
			x86_64)
				echo amd64
				return 0
				;;
			arm | armv[0-9]*)
				echo arm
				return 0
				;;
			aarch64)
				echo arm64
				return 0
				;;
			mips*)
				echo >&2 "I see you are running on mips but I don't know how to determine endianness yet, so I cannot select a correct arch to fetch."
				echo >&2 "Consider installing \"go\" on the system which I can use to determine the correct arch or specify it explictly by setting TARGETARCH"
                rm -rf $dir
				exit 1
				;;
			*)
				echo "${uArch}"
				return 0
				;;
		esac

	fi

	# default value
	echo >&2 "Unable to determine CPU arch, falling back to amd64. You can specify a target arch by setting TARGETARCH"
	echo amd64
}

imageTag="$1"
image="${imageTag%%[:@]*}"
imageTag="${imageTag#*:}"

if [[ "$image" == "$imageTag" ]]; then
    tag="latest"
    imageTag="latest"

fi

digest="${imageTag##*@}"
tag="${imageTag%%@*}"

# add prefix library if passed official image
if [[ "$image" != *"/"* ]]; then
    image="library/$image"
fi

imageFile="${image//\//_}" # "/" can't be in filenames :)

if [ -z "$DOCKER_TOKEN" ]; then
    DOCKER_TOKEN="$(curl -fsSL "$AUTH_BASE/token?service=$AUTH_SERVICE&scope=repository:$image:pull" | jq --raw-output '.token')"
fi

manifestJson="$(
    curl -fsSL \
        -H "$AUTH_HEADER $DOCKER_TOKEN" \
        -H 'Accept: application/vnd.docker.distribution.manifest.v2+json' \
        -H 'Accept: application/vnd.docker.distribution.manifest.list.v2+json' \
        -H 'Accept: application/vnd.docker.distribution.manifest.v1+json' \
        "$REGISTRY_BASE/v2/$image/manifests/$digest"
)"
if [ "${manifestJson:0:1}" != '{' ]; then
    echo >&2 "error: /v2/$image/manifests/$digest returned something unexpected:"
    echo >&2 "  $manifestJson"
    rm -rf $dir
    exit 1
fi

imageIdentifier="$image:$tag@$digest"

schemaVersion="$(echo "$manifestJson" | jq --raw-output '.schemaVersion')"
case "$schemaVersion" in
    2)
        mediaType="$(echo "$manifestJson" | jq --raw-output '.mediaType')"

        case "$mediaType" in
            application/vnd.docker.distribution.manifest.v2+json)
                handle_single_manifest_v2 "$manifestJson"
                ;;
            application/vnd.docker.distribution.manifest.list.v2+json)
                layersFs="$(echo "$manifestJson" | jq --raw-output --compact-output '.manifests[]')"
                IFS="$newlineIFS"
                mapfile -t layers <<< "$layersFs"
                unset IFS

                found=""
                targetArch="$(get_target_arch)"
                # parse first level multi-arch manifest
                for i in "${!layers[@]}"; do
                    layerMeta="${layers[$i]}"
                    maniArch="$(echo "$layerMeta" | jq --raw-output '.platform.architecture')"
                    if [ "$maniArch" = "${targetArch}" ]; then
                        digest="$(echo "$layerMeta" | jq --raw-output '.digest')"
                        # get second level single manifest
                        submanifestJson="$(
                            curl -fsSL \
                                -H "$AUTH_HEADER $DOCKER_TOKEN" \
                                -H 'Accept: application/vnd.docker.distribution.manifest.v2+json' \
                                -H 'Accept: application/vnd.docker.distribution.manifest.list.v2+json' \
                                -H 'Accept: application/vnd.docker.distribution.manifest.v1+json' \
                                "$REGISTRY_BASE/v2/$image/manifests/$digest"
                        )"
                        handle_single_manifest_v2 "$submanifestJson"
                        found="found"
                        break
                    fi
                done
                if [ -z "$found" ]; then
                    echo >&2 "error: manifest for $maniArch is not found"
                    rm -rf $dir
                    exit 1
                fi
                ;;
            *)
                echo >&2 "error: unknown manifest mediaType ($imageIdentifier): '$mediaType'"
                rm -rf $dir
                exit 1
                ;;
        esac
        ;;

    1)

        layersFs="$(echo "$manifestJson" | jq --raw-output '.fsLayers | .[] | .blobSum')"
        IFS="$newlineIFS"
        mapfile -t layers <<< "$layersFs"
        unset IFS

        history="$(echo "$manifestJson" | jq '.history | [.[] | .v1Compatibility]')"
        imageId="$(echo "$history" | jq --raw-output '.[0]' | jq --raw-output '.id')"

        echo "Downloading '$imageIdentifier' (${#layers[@]} layers)..."
        for i in "${!layers[@]}"; do
            echo $(($i+1))/"${#layers[@]}"
            imageJson="$(echo "$history" | jq --raw-output ".[${i}]")"
            layerId="$(echo "$imageJson" | jq --raw-output '.id')"
            imageLayer="${layers[$i]}"

            mkdir -p "$dir/$layerId"

            # TODO figure out why "-C -" doesn't work here
            # "curl: (33) HTTP server doesn't seem to support byte ranges. Cannot resume."
            # "HTTP/1.1 416 Requested Range Not Satisfiable"
            if [ -f "$dir/$layerId/layer.tar" ]; then
                # TODO hackpatch for no -C support :'(
                echo "skipping existing ${layerId:0:12}"
                continue
            fi
            if [ -z "$DOCKER_TOKEN" ]; then
                DOCKER_TOKEN="$(curl -fsSL "$AUTH_BASE/token?service=$AUTH_SERVICE&scope=repository:$image:pull" | jq --raw-output '.token')"
            fi
            fetch_blob "$DOCKER_TOKEN" "$image" "$imageLayer" "$dir/$layerId/layer.tar" --progress-bar
        done
        ;;

    *)
        echo >&2 "error: unknown manifest schemaVersion ($imageIdentifier): '$schemaVersion'"
        rm -rf $dir
        exit 1
        ;;
esac

echo

echo "Download of image into '$dir' complete."

sudo portoctl build $layers_paths -o $result_layer -M virt_mode=app net=inherited
sudo chown $(id -u):$(id -g) $result_layer

rm -rf $dir

echo "Use something like the following to run porto container with result layer:"
echo "  portoctl exec -L ./$result_layer abc command=bash"

if [ -n "$run" ]; then
    sudo portoctl exec -L ./$result_layer abc command=bash
fi
