#!/usr/bin/python

import os
import yaml
import time
from yaml import load,dump
import subprocess

CONFIG="qemu_launcher.conf"
NETCONF_PATH="netconf"
QEMU_IMG="./qemu-img"
QEMU="./qemu-system-x86_64"
RESPAWN_TIMEOUT=10

# Config example:
#
# vcpu_count : 0
# mem_size : 1G
# disk_size : 4G
# disk_img_path : "xenial.img"
# password : "passw0rd"
# hostname : "max7255-vm"
# netconf_path : "netconf"

USER_DATA_TEMPLATE = """\
#cloud-config
password: {}
chpasswd: {{ expire: False }}
ssh_pwauth: True"""

META_DATA_TEMPLATE = """\
instance-id: iid-{}
local-hostname: {}
network-interfaces: |
{}
growpart:
   mode: auto
   devices: ['/']
resize_rootfs: True"""

def create_qcow2(path, src_path, size):
    qemu_img_args = (QEMU_IMG, "create", "-f", "qcow2", "-b", src_path, path, size)

    print "qemu-img args: {}".format(qemu_img_args)

    subprocess.check_call(qemu_img_args)

def mac2ll(mac_str):
    mac = [int(i, 16) for i in mac_str.split(":")]

    return "fe80::{:02x}{:02x}:{:02x}ff:fe{:02x}:{:02x}{:02x}"\
            .format(mac[0] ^ 2, mac[1],mac[2],mac[3],mac[4],mac[5])

class InstanceCtx :
    def __init__(self, env):
        if env is not None:
            self.BasePort = int(env["BSCONFIG_IPORT"])
            self.Name = env["BSCONFIG_INAME"]

            self.MonPort = self.BasePort
            self.SerialPort = self.BasePort + 1
            self.SshPort = self.BasePort + 2
            self.HttpPort = self.BasePort + 3
        else:
            self.BasePort = 0

ENI_ENTRY_TEMPLATE="""\
   auto {}
   iface {} inet6 static
   gateway {}
{}

"""

USER_ENI="""\
   auto eth0
   iface eth0 inet dhcp"""

class VmNetInterface :
    def GetENIDesc(name):
        addrs = ""

        for addr in self.Addrsv6:
            addrs += "address {}/128\n".format(addr)

        return ENI_ENTRY_TEMPLATE.format(name, name, self.HostLlAddr, addrs)

    def GetCINetV1Dict(name):
        entry = {}

        entry["mac_address"] = self.MacAddr
        entry["name"] = name
        entry["type"] = "physical"

        entry["subnets"] = []

        for addr in self.Addrsv6:
            subnet = {}
            subnet["type"] = "static"
            subnet["gateway"] = self.HostLlAddr
            subnet["address"] = addr

            entry["subnets"] += [subnet]

    def __init__(self, config):
        if config is not None:
            self.Name = config["name"]
            self.Addrsv6 = config["addrs_v6"]
            if "mac_addr" in config:
                self.MacAddr = config["mac_addr"]
            else:
                self.MacAddr = "52:54:00:12:34:56"

            self.HostLlAddr = mac2ll(open("/sys/class/net/{}/address"\
                                          .format(self.Name).read()))

        else:
            self.Name = ""
            self.Addrsv6 = []
            self.MacAddr = ""

class Vm :
    VCpuCount = 0
    MemSize = 0
    DiskSize = 0
    DiskImgPath = "current.img"
    DiskQcow2Path = "current.qcow2"
    Ifaces = []
    Password = "P@ssw0rd"
    Hostname = "qemu-vm"
    NetconfPath = NETCONF_PATH
    HasKvm = os.access("/dev/kvm", os.R_OK | os.W_OK)
    HasUserNet = True

    def __init__(self):
        if not os.path.exists(self.NetconfPath):
            os.mkdir(self.NetconfPath)

    def ParseConfig(self, filename):
        config = load(open(filename, "r").read())

        self.VCpuCount = int(config["vcpu_count"])
        self.MemSize = config["mem_size"]
        self.DiskSize = config["disk_size"]
        self.DiskImgPath = config["disk_img_path"]

        if os.access("/dev/net/tun", os.R_OK | os.W_OK) and "ifaces" in config:
            HasUserNet = False
            for iface in config["ifaces"]:
                self.Ifaces += VmNetInterface(iface)

        if "password" in config :
            self.Password = config["password"]

        if "hostname" in config :
            self.Hostname = config["hostname"]

        if "netconf_path" in config:
            self.NetconfPath = config["netconf_path"]

    def WriteCIConfig(self, ctx):
        with open(self.NetconfPath + "/meta-data", "w") as f:
            if self.HasUserNet:
                meta_net_data = USER_ENI
            else:
                meta_net_data = ""

                iface_count = 0
                for iface in self.Ifaces:
                    meta_net_data += iface.GetENIDesc("eth{}".format(iface))
                    iface_count += 1

            f.write(META_DATA_TEMPLATE.format(ctx.Name, self.Hostname, meta_net_data))

        with open(self.NetconfPath + "/user-data", "w") as f:
            f.write(USER_DATA_TEMPLATE.format(self.Password))

        with open(self.NetconfPath + "/network-config", "w") as f:
            net_config = {}
            net_config["version"] = 1
            net_config["config"] = []

            if self.HasUserNet:
                user_net = {}
                user_net["mac_address"] = "52:54:00:12:34:56"
                user_net["name"] = "eth0"
                user_net["subnets"] = [{"type" : "dhcp"}]
                user_net["type"] = "physical"

                net_config["config"] = [user_net]

            else:
                iface_count = 0

                for iface in self.Ifaces:
                    net_config["config"] += [iface.GetCINetV1Dict("eth{}".format(iface_count))]

            f.write(dump(net_config))

    def GetQemuCmdline(self, ctx):
        cmdline = []

        if self.HasKvm:
            cmdline += ["--enable-kvm"]

        cmdline += ["-m", str(self.MemSize)]
        cmdline += ["-smp", str(self.VCpuCount)]
        cmdline += ["-nographic"]

        #mon
        cmdline += ["-chardev", "socket,id=monsk,host={},port={},server,nowait"\
                    .format("::", ctx.MonPort)]
        cmdline += ["-mon", "monsk"]
        cmdline += ["-monitor", "none"]


        cmdline += ["-chardev", "socket,id=serialsk,host={},port={},server,nowait"\
                    .format("::", ctx.SerialPort)]
        cmdline += ["-serial", "chardev:serialsk"]

        #FIXME: tap net

        cmdline += ["-drive", "file={},if=virtio,cache=none".format(self.DiskQcow2Path)]
        cmdline += ["-drive", "file=fat:{},if=virtio,file.label=cidata,readonly=on".format(self.NetconfPath)]

        if self.HasUserNet:
            cmdline += ["-netdev", "user,id=qemu_net,hostfwd=tcp::{}-:22,"\
                                   "hostfwd=tcp::{}-:80"\
                                   .format(ctx.SshPort, ctx.HttpPort)]
            cmdline += ["-device", "virtio-net-pci,netdev=qemu_net"]

        return cmdline

def main():
    ctx = InstanceCtx(os.environ)

    vm = Vm()
    vm.ParseConfig(CONFIG)

    vm.WriteCIConfig(ctx)

    while True:
        print "Spawn qemu..."

        try:
            create_qcow2(vm.DiskQcow2Path, vm.DiskImgPath, vm.DiskSize)

            qemu_args = tuple([QEMU] + vm.GetQemuCmdline(ctx))

            print "qemu args: {}".format(qemu_args)

            subprocess.check_call(qemu_args)
        except BaseException as e:
            print "Got exception: {}".format(e)

        print "Qemu exited..."

        time.sleep(RESPAWN_TIMEOUT)


if __name__ == '__main__':
    main()
