#include <string.h>
#include <err.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/prctl.h>
#include <sys/select.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <fcntl.h>

extern void setproctitle_init(int argc, char *argv[], char *envp[]);
extern void setproctitle(const char *fmt, ...);
extern int setup_seccomp();

static int status_read_fd;

#define SETSIG(sa, sig, fun, flags) { \
    memset(&sa, 0, sizeof(sa)); \
    sa.sa_handler = fun; \
    sa.sa_flags = flags; \
    sigemptyset(&sa.sa_mask); \
    sigaction(sig, &sa, NULL); \
}

// dist knows nothing about this constant, dunno why
#ifndef PR_SET_CHILD_SUBREAPER
#define PR_SET_CHILD_SUBREAPER 36
#endif

void on_child_exited(int sig) {
    (void)sig; // unused
    int wstatus;
    int ret = 0;

    fd_set rfds;
    FD_ZERO(&rfds);
    FD_SET(status_read_fd, &rfds);
    struct timeval tv = { .tv_sec = 0, .tv_usec = 0};
    if (select(status_read_fd + 1, &rfds, NULL, NULL, &tv) > 0) {
        if (read(status_read_fd, &wstatus, sizeof(wstatus)) == sizeof(wstatus)) {
            if (WIFEXITED(wstatus))
                ret = WEXITSTATUS(wstatus);
            else if (WIFSIGNALED(wstatus))
                ret = -WTERMSIG(wstatus);
        }
    }

    exit(ret);
}

void on_signal(int sig) {
    kill(0, sig);
}

int child(pid_t parent_pid, int status_write_fd, char **argv) {
    struct sigaction sa;
    for(int f = 1; f < NSIG; f++)
        SETSIG(sa, f, SIG_IGN, SA_RESTART);

    SETSIG(sa, SIGSEGV, on_signal, SA_RESTART|SA_RESETHAND);
    SETSIG(sa, SIGABRT, on_signal, SA_RESTART|SA_RESETHAND);
    SETSIG(sa, SIGINT, on_signal, SA_RESTART|SA_RESETHAND);
    SETSIG(sa, SIGTERM, on_signal, SA_RESTART|SA_RESETHAND);
    SETSIG(sa, SIGCHLD, SIG_DFL, SA_RESTART);

    if (prctl(PR_SET_CHILD_SUBREAPER, 1) == -1)
        errx(5, "Failed to become subreaper");

    if (setpgrp() == -1)
        errx(6, "Failed to setpgrp");

    // we do not check for result here since the session may be non-interactive (with no terminal)
    tcsetpgrp(0, getpgid(0));

    pid_t child_pid = fork();
    if (child_pid < 0)
        errx(7, "Fork failed");
    else if (child_pid == 0) {
        for(int f = 1; f < NSIG; f++)
            SETSIG(sa, f, SIG_DFL, SA_RESTART);
        return execvp(argv[0], argv);
    }

    chdir("/");
    close(0);
    close(1);
    close(2);

    int wstatus, ret = 0;
    pid_t pid;
    while(1) {
        pid = wait(&wstatus);
        if (pid == child_pid) {
            if (WIFEXITED(wstatus))
                ret = WEXITSTATUS(wstatus);
            else if (WIFSIGNALED(wstatus))
                ret = -WTERMSIG(wstatus);

            write(status_write_fd, &wstatus, sizeof(wstatus));
            kill(parent_pid, SIGUSR1);
        }
        if (pid == -1 && errno == ECHILD)
            break;
    }
    return ret;
}

int main(int argc, char **argv, char **envp) {
    setproctitle_init(argc, argv, envp);
    if (argc < 3)
        errx(1, "Usage: %s <username> <command>...\n", argv[0]);

    char *session_id = getenv("PORTOSHELL_SESSION_ID");
    if (!session_id)
        session_id = (char*)"unknown";

    setproctitle("portoshell-session [user=%s] [session=%s]", argv[1], session_id);
    if (setup_seccomp() == -1)
        errx(2, "Setup seccomp failed");

    struct sigaction sa;
    SETSIG(sa, SIGUSR1, on_child_exited, SA_RESTART);
    pid_t parent_pid = getpid();

    int status_pipe[2];
    if (pipe2(status_pipe, O_CLOEXEC | O_DIRECT | O_NONBLOCK) == -1)
        errx(8, "Failed to create status pipe pair");

    pid_t child_pid = fork();
    if (child_pid < 0)
        errx(3, "Master fork failed");
    else if (child_pid == 0) {
        close(status_pipe[0]);
        int ret;
        if ((ret = child(parent_pid, status_pipe[1], argv + 2)) == -1)
            errx(4, "Child fork failed");
        return ret;
    }

    chdir("/");
    close(0);
    close(1);
    close(2);
    close(status_pipe[1]);
    status_read_fd = status_pipe[0];

    int wstatus;
    while(1) {
        pid_t pid = waitpid(child_pid, &wstatus, 0);
        if (pid == -1 && errno == ECHILD)
            return 0;
        if (pid != child_pid)
            continue;
        if (WIFEXITED(wstatus))
            return WEXITSTATUS(wstatus);
        if (WIFSIGNALED(wstatus))
            return -WTERMSIG(wstatus);
    }
    return 0;
}
