try:
    import syslog
except ImportError:
    syslog = None


class AuthSyslog(object):
    DEBUG = "LOG_DEBUG"
    INFO = "LOG_INFO"
    NOTICE = "LOG_NOTICE"
    WARNING = "LOG_WARNING"
    ERROR = "LOG_ERR"
    CRITICAL = "LOG_CRITICAL"
    ALERT = "LOG_ALERT"
    EMERGENCY = "LOG_EMERG"

    def __init__(self, name):
        self.__initialized = None
        self.name = name

    def open(self):
        if self.__initialized is not None:
            return

        if syslog is None:
            self.__initialized = False
            return

        for attempt in xrange(3):
            try:
                syslog.openlog(self.name, syslog.LOG_PID, syslog.LOG_AUTH)
                self.__initialized = True
                break
            except EnvironmentError:
                pass
        if self.__initialized is None:
            self.__initialized = False

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        if self.__initialized:
            self.close()

    def log(self, level, message):
        if self.__initialized:
            try:
                level = getattr(syslog, level)
            except AttributeError:
                return

            try:
                syslog.syslog(level, message)
            except EnvironmentError:
                pass

    def debug(self, message):
        self.log(AuthSyslog.DEBUG, message)

    def info(self, message):
        self.log(AuthSyslog.INFO, message)

    def notice(self, message):
        self.log(AuthSyslog.NOTICE, message)

    def warning(self, message):
        self.log(AuthSyslog.WARNING, message)

    warn = warning

    def error(self, message):
        self.log(AuthSyslog.ERROR, message)

    def critical(self, message):
        self.log(AuthSyslog.CRITICAL, message)

    def alert(self, message):
        self.log(AuthSyslog.ALERT, message)

    def emergency(self, message):
        self.log(AuthSyslog.EMERGENCY, message)

    def close(self):
        if self.__initialized:
            try:
                syslog.closelog()
            except EnvironmentError:
                pass
        self.__initialized = None
