# cython: language_level=2

from cpython.exc cimport PyErr_SetFromErrno
from posix.stdlib cimport ptsname
from posix.time cimport timeval
from posix.fcntl cimport open
from posix.unistd cimport close, fchdir

import contextlib


cdef extern from "<fcntl.h>" nogil:
    cdef int O_PATH
    cdef int O_DIRECTORY
    cdef int O_CLOEXEC
    cdef int O_NOFOLLOW

    int openat(int, const char *, int, ...)


cdef extern from "<time.h>" nogil:
    int futimes(int fd, timeval tv[2])


cpdef public futime(object fileobject, object times) with gil:
    cdef timeval tv[2]

    if not isinstance(fileobject, int):
        fileobject = fileobject.fileno()

    if times is None:
        ret = futimes(fileobject, NULL)
    else:
        tv[0].tv_sec = times[0]
        tv[0].tv_usec = 0
        tv[1].tv_sec = times[1]
        tv[1].tv_usec = 0
        ret = futimes(fileobject, tv)

    if ret != 0:
        PyErr_SetFromErrno(OSError)


cpdef public py_ptsname(int fd) with gil:
    ret = ptsname(fd)
    if ret is None:
        PyErr_SetFromErrno(OSError)
    else:
        return <bytes>ret


@contextlib.contextmanager
def safe_chdir(str target):
    current_dir_fd = open(".", O_CLOEXEC | O_DIRECTORY | O_PATH)
    if current_dir_fd == -1:
        PyErr_SetFromErrno(OSError)
        return

    try:
        target_parent, target_child = target.rsplit('/', 1)
        target_parent_fd = open(target_parent, O_CLOEXEC | O_DIRECTORY | O_PATH)
        if target_parent_fd == -1:
            PyErr_SetFromErrno(OSError)
            return
        try:
            if fchdir(target_parent_fd) == -1:
                PyErr_SetFromErrno(OSError)
                return

            current_fd = openat(target_parent_fd,
                                target_child,
                                O_CLOEXEC | O_DIRECTORY | O_PATH | O_NOFOLLOW)
            if current_fd == -1:
                PyErr_SetFromErrno(OSError)
                return
            if fchdir(current_fd) == -1:
                PyErr_SetFromErrno(OSError)
                return
            try:
                yield
            finally:
                close(current_fd)

        finally:
            close(target_parent_fd)
    finally:
        fchdir(current_dir_fd)
        close(current_dir_fd)
