import re

from ya.skynet.util import logging


def resolve_in_config(cfg, value):
    matches = re.finditer('\$\{(.*?)\}', value)

    if not matches:
        return value

    for match in matches:
        cfg_value = cfg
        for sub in match.group(1).split('.'):
            cfg_value = getattr(cfg_value, sub)

        value = value[:match.start()] + cfg_value + value[match.end():]

    return value


def find_object(name):
    if hasattr(logging, name):
        return getattr(logging, name)

    if '.' not in name:
        raise Exception('Cant find object {0!r}'.format(name))

    module, class_name = name.rsplit('.', 1)
    return getattr(__import__(module, fromlist=['']), class_name)


def create_formatters(conf):
    formatters = {}
    for name, props in conf.items():
        if '_class' in props:
            cls_mod, cls_name = props.pop('_class').rsplit('.', 1)
            mod = __import__(cls_mod, fromlist=[cls_name])
            cls = getattr(mod, cls_name)
        else:
            cls = logging.Formatter      # pylint: disable=E1101
        formatters[name] = cls(**props)  # pylint: disable=W0142
    return formatters


def create_loggers(conf, handlers):
    loggers = {}
    for name, props in conf.items():
        logger = logging.getLogger(name)  # pylint: disable=E1101
        for handler_name in props.get('handlers', ()):
            logger.addHandler(handlers[handler_name])
        if 'level' in props:
            logger.setLevel(getattr(logging, props['level']))

        loggers[name] = logger
    return loggers


def create_handlers(cfg, log_conf, formatters):
    handlers = {}
    for name, props in log_conf.items():
        handler_class_name = props['class']

        handler_class = find_object(handler_class_name)

        if 'args' in props:
            args = []
            for arg in props['args']:
                args.append(find_object(arg))
        else:
            args = []

        kwargs = props.pop('kwargs', {})
        for key, value in kwargs.items():
            kwargs[key] = resolve_in_config(cfg, value)

        handler = handler_class(*args, **kwargs)  # pylint: disable=W0142

        if 'level' in props:
            handler.setLevel(getattr(logging, props['level']))

        if 'formatter' in props:
            formatter = formatters[props['formatter']]
            handler.setFormatter(formatter)

        handlers[name] = handler
    return handlers


def setup_logging(cfg):
    log_conf = cfg['logging']
    logging.renameLevels(log_conf.get('levelNameScheme', None))

    formatters = create_formatters(log_conf['formatters'])
    handlers = create_handlers(cfg, log_conf['handlers'], formatters)

    create_loggers(log_conf['loggers'], handlers)
