from __future__ import absolute_import

import os
import shutil
import tempfile

from ya.skynet.util.functional import memoized

from .sysutils import ensure_dir

from porto.exceptions import (
    Busy,
    InvalidValue,
    LayerNotFound,
    VolumeAlreadyExists,
    VolumeAlreadyLinked,
    VolumeNotLinked,
    VolumeNotFound,
)
from porto.api import Connection


LAYER = b'portoshell-utils'
MOUNTPOINT = b'portoshell_utils'


@memoized
def get_portoconn(main=False):
    return Connection()


@memoized
def get_container_root(container_name):
    return get_portoconn().ConvertPath("/", container_name, "self")


# may be we should always use this root? do we need any special capabilities?
@memoized
def get_container_proc_root(container_name):
    pid = get_portoconn().GetProperty(container_name, 'root_pid')
    return '/proc/%s/root/' % (pid,)


def find_in_container(container_name, candidates, fallback, isdir=False):
    conn = get_portoconn()
    check = os.path.isdir if isdir else os.path.isfile
    for candidate in candidates:
        path = conn.ConvertPath(candidate, container_name, "self")
        if os.path.exists(path) and check(path):
            return candidate

    return fallback


def make_utils_volume(container_name, tarball_path):
    conn = get_portoconn()
    try:
        try:
            layer = conn.FindLayer(LAYER)
        except LayerNotFound:
            layer = conn.MergeLayer(LAYER, tarball_path, private_value=tarball_path)
        else:
            private = layer.GetPrivate()
            if private != tarball_path:
                try:
                    layer.Merge(tarball_path, private_value=tarball_path)
                except Busy:
                    pass
    except InvalidValue:
        return

    root = get_container_root(container_name)
    dest = os.path.join(root, MOUNTPOINT)
    ensure_dir(dest)

    try:
        volume = conn.CreateVolume(dest, [layer], read_only="true")
    except VolumeAlreadyExists:
        volume = conn.FindVolume(dest)

    try:
        volume.Link(container_name)
    except VolumeAlreadyLinked:
        pass

    try:
        volume.Unlink("self")
    except VolumeNotLinked:
        pass


def make_home_volume(container_name, extra_files):
    conn = get_portoconn()
    home_root = get_container_root(container_name)

    for home_dir, retriable in (
        (os.path.join(home_root, 'portoshell_homes'), True),
        (home_root, False),
    ):
        home_tmp = tempfile.mkdtemp(prefix='home', dir=home_dir)
        home = conn.CreateVolume(home_tmp, backend='tmpfs', space_limit='50M')
        try:
            home.Link(container_name)
            extra_files = extra_files or {}
            for filename, data in extra_files.iteritems():
                fullpath = os.path.abspath(os.path.join(home_tmp, filename))
                if not fullpath.startswith(home_tmp + '/') or '/' in filename:
                    raise RuntimeError("Invalid filename: %r, possible break-in attempt" % (filename,))

                temp_fd, temp_filename = tempfile.mkstemp(dir=home_tmp)
                with os.fdopen(temp_fd, 'wb') as f:
                    f.write(data)
                os.rename(temp_filename, fullpath)
        except:
            shutil.rmtree(home_tmp, ignore_errors=True)
            if retriable:
                continue
            else:
                raise
        finally:
            home.Unlink('self')

        return home_tmp, conn.ConvertPath(home.path, 'self', container_name)


def make_yatool_volume(container_name):
    yatool_path = '/place/ya_tool_cache'
    bind_path = '/ya_tool_cache'
    private_value = 'PORTOSHELL yatool'

    if not os.path.exists(yatool_path) or not os.path.isdir(yatool_path):
        return

    conn = get_portoconn()
    root = get_container_root(container_name)
    volume_path = os.path.join(root, 'ya_tool_cache')

    try:
        volume = conn.FindVolume(volume_path)
    except VolumeNotFound:
        ensure_dir(volume_path)
        volume = conn.CreateVolume(
            path=volume_path,
            layers=[yatool_path],
            backend='overlay',
            space_limit='5G',
            private_value=private_value,
        )

    try:
        volume.Link(container_name, bind_path)
    except Busy:  # Volume appears to be already linked to the same place
        volume.Link(container_name)
    finally:
        try:
            volume.Unlink('self')
        except VolumeNotLinked:
            pass

    return True
