from __future__ import absolute_import

import os
import sys
import uuid
import errno
import socket
import signal
from SocketServer import BaseRequestHandler

from ya.skynet.util.net.socketstream import SocketStream

from .slots.slot_sources import find_slot
from .slots.exceptions import AuthError, SlotLookupError, ConfigurationLookupError
from .authenticate import authenticate, format_key
from .shell import Context, StartupException
from .authsyslog import AuthSyslog
from .utils import suppress
from .heartbeat import schedule_report
from . import logger

import msgpack


class RawHandler(BaseRequestHandler):
    def _send_error(self, exc):
        if self.ctx.api_mode:
            SocketStream(self.request).writeBEStr(msgpack.dumps({'error': True, 'message': str(exc)}))
        else:
            self.request.sendall(str(exc))

    def handle(self):
        signal.alarm(60)
        log = logger.logging.MessageAdapter(
            self.server.logger.getChild('conn'),
            fmt='[%(pid)s] %(message)s',
            data={'pid': os.getpid()},
        )

        stream = SocketStream(self.request)
        log.debug('created stream, trying to read msg')
        ctx = self.ctx = Context(log)
        session = None
        auth_key = None
        slot = None
        try:
            msg = stream.readBEStr()
            log.debug('read msg len %d', len(msg))
            try:
                msg = msgpack.loads(msg)
                log.debug('loaded msg')
                token = msg['token']
                # FIXME should accept variable lookup parameters set (without 'slot', for yp)
                token['user'] = msg['user']
                token['slot'] = msg['slot']
                token['configuration_id'] = msg.get('configuration_id', '')
                ctx.api_mode = msg.get('api_mode', False)

                login_user = msg['user'] or token['acc_user']
                log.info("user %s@%s trying to enter %r / %r as `%s`",
                         token['acc_user'],
                         token['acc_host'],
                         msg['slot'],
                         msg.get('configuration_id'),
                         login_user)

                slot = find_slot(slot_name=token['slot'],
                                 configuration_id=token['configuration_id'],
                                 iss_enabled=self.server.iss)

                auth_key = authenticate(log.getChild('auth'),
                                        user=login_user,
                                        slot_info=slot,
                                        token=token,
                                        signs=msg['signs'],
                                        keys_storage=self.server.keys_storage,
                                        ca_storage=self.server.ca_storage,
                                        ) if self.server.check_auth else None
                token['acc_user'] = next(iter(auth_key.userNames), None)

                ctx.sessionleader_session_id = str(uuid.uuid4())
                ctx.sessionleader_user = token['acc_user']
                session = ctx.make_session(0)
                session.container_name = "{}/ps-{}".format(slot.container, os.getpid())
                log.info("joining container %s", session.container_name)

                ctx.user = msg['user']
                ctx.unset_env = msg.get('unset_env')
                ctx.interactive_cmd = msg.get('interactive_cmd', False)
                ctx.streaming = msg.get('streaming', True)
                ctx.tools_tarball = self.server.tools_tarball
                ctx.telnet_timeout = msg.get('telnet_timeout')
                ctx.extra_files = msg.get('files')
                ctx.setup_motd(slot, self.server.hostname)

                session.extra_env = msg.get('extra_env') or []
                session.cwd = slot.instance_dir
                session.cmd = msg.get('command')
                session.tag = '%s@%s' % (token['acc_user'], token['acc_host'])

                for env_key, env_val in slot.get_env_vars():
                    session.extra_env.insert(0, (env_key, env_val))
                server = session.start_server(sock=self.request)
                signal.alarm(0)
            except AuthError as e:
                with AuthSyslog('portoshell') as syslog:
                    syslog.info(
                        "%s@%s failed authentication as %s to %s: %s" % (
                            token['acc_user'],
                            token['acc_host'],
                            login_user,
                            slot.as_auth_info(),
                            e))
                log.warning('auth failure: %s', e)
                self._send_error('Auth failure: %s' % (e,))
            except (SlotLookupError, ConfigurationLookupError) as e:
                log.warning('not found slot %r, configuration_id %r for %s@%s' % (token['slot'],
                                                                                  token['configuration_id'],
                                                                                  token['acc_user'],
                                                                                  token['acc_host']))
                self._send_error(e)
            except StartupException as e:
                log.exception('shell start failed: %s', e.msg, exc_info=sys.exc_info())
                self._send_error("Shell start failed: %s" % (e.msg,))
            except Exception as e:
                log.exception('shell start failed: %s', e, exc_info=sys.exc_info())
                self._send_error("Shell start failed: %s" % (e,))
            else:
                schedule_report(
                    log,
                    transport='raw',
                    user=login_user,
                    acc_user=token['acc_user'],
                    acc_host=token['acc_host'],
                    streaming=msg.get('streaming'),
                    timeout=msg.get('telnet_timeout'),
                    interactive_cmd=msg.get('interactive_cmd'),
                    api_mode=msg.get('api_mode'),
                    extra_files=list((msg.get('files') or {}).keys()),
                    command=msg.get('command'),
                    auth_type=auth_key and type(auth_key).__name__,
                    auth_bits=auth_key and auth_key.get_bits(),
                    slot_type=type(slot).__name__,
                    slot_info=slot.as_auth_info(),
                )
                with AuthSyslog('portoshell') as syslog:
                    syslog.info(
                        "%s@%s authenticated as %s by %s into %s, session_id=%s" % (
                            token['acc_user'],
                            token['acc_host'],
                            login_user,
                            format_key(auth_key),
                            slot.as_auth_info(),
                            ctx.sessionleader_session_id,
                        ))
                server.run()

        except socket.error as e:
            if e.errno not in (errno.ECONNRESET, errno.EPIPE, errno.ECONNABORTED):
                raise
        finally:
            if session and session.container is not None:
                suppress(session.container.Destroy)

            ctx.close_session(0)

            try:
                self.request.shutdown(socket.SHUT_RDWR)
            except EnvironmentError as e:
                if e.errno != errno.ENOTCONN:
                    log.warning("failed to shutdown connection to client: %s", e)

            self.request.close()
