import os
import re
import sys
import base64

from itertools import groupby
from collections import namedtuple

from yt import yson
from .slot import NannySlot, YpLiteSlot, QloudSlot, normalize_ip, box_type_attribute_path
from .yp_fetcher import get_acl

from infra.skylib import http_tools

from ya.skynet.util import logging
from ya.skynet.util.functional import cached


__all__ = ['fetch_slots']


if sys.version_info.major < 3:
    def b(s):
        if s is None:
            return None
        return s.encode('utf-8') if isinstance(s, unicode) else str(s)  # noqa
else:
    def b(s):
        if s is None:
            return None
        return s if isinstance(s, bytes) else bytes(str(s), 'utf-8')


YasmTags = namedtuple('YasmTags', ('ctype', 'itype', 'prj'))


def _get_meta_container_path(options, log):
    HOOK_SUFFIX = '/iss_hook_start'
    container = options.get('metaContainerPath')

    if not container:  # backward compatibility
        container = options['containerPath']
        if container.endswith(HOOK_SUFFIX):
            container = container[:-len(HOOK_SUFFIX)]
        log.debug('deduced meta container path: %s', container)
    else:
        log.debug('read meta container path: %s', container)

    return container


def _get_mtn_interfaces(properties, log):
    mtn_interfaces = properties.get('container/constraints/slot.ip')
    if mtn_interfaces:
        mtn_interfaces = mtn_interfaces.split(';')
        mtn_interfaces = [b(normalize_ip(ip.strip().split(' ', 1)[1])) for ip in mtn_interfaces]
        log.debug('read mtn interfaces from "slot.ip": %s', mtn_interfaces)
    else:
        mtn_interfaces = properties.get('properties/BACKBONE_IP_ADDRESS')
        if mtn_interfaces:
            mtn_interfaces = [b(normalize_ip(mtn_interfaces))]
            log.debug('read mtn interfaces from "BACKBONE_IP_ADDRESS": %s', mtn_interfaces)

    if not mtn_interfaces:
        mtn_interfaces = None
        log.debug('got no mtn interfaces')

    return mtn_interfaces


def _get_mtn_slot_container(options, container, log):
    mtn_slot_container = options.get('slotContainer')
    if not mtn_slot_container:
        mtn_slot_container = None if '/' not in container else container.split('/', 1)[0]
        mtn_slot_container = (mtn_slot_container
                              if mtn_slot_container and re.match(r'^ISS-AGENT--[a-zA-Z0-9]+$', mtn_slot_container)
                              else None)
        log.debug('deduced mtn slot container: %s', mtn_slot_container)
    else:
        log.debug('read mtn slot container from "slotContainer": %s', mtn_slot_container)

    return mtn_slot_container


def _get_item(dictionary, item, log, optional=False, default=None):
    try:
        result = dictionary.get(item, default) if optional else dictionary[item]
        log.debug('read %s: %s', item, result)
        return result
    except KeyError:
        log.debug('got no %s')
        raise


def _get_project(properties, log):
    soa = 'properties/all-tags' if 'properties/all-tags' in properties else 'properties/tags'

    tags = properties.get(soa, '').split()
    project = next((tag for tag in tags if tag.startswith('a_prj_')), '')[6:]
    log.debug('read project from "%s": %s', soa, project)
    return project


def _get_yasm_labels(properties, log):
    ctype = _get_item(properties, 'properties/INSTANCE_TAG_CTYPE', log, optional=True, default='none')
    itype = _get_item(properties, 'properties/INSTANCE_TAG_ITYPE', log, optional=True, default='none')
    prj = _get_item(properties, 'properties/INSTANCE_TAG_PRJ', log, optional=True, default='none')

    return YasmTags(ctype, itype, prj)


def _fetch_yp_lite_pods_infos(log):
    pod_id_to_infos_index = {}

    for options in http_tools.fetch_json("http://localhost:25536/pods/info", "YP", "pods", log=log).get("pod_info", []):
        pod = options.get('id')
        if pod is None:
            continue

        info = {'labels': {}}
        try:
            yp_hard_flag = _get_item(options, 'has_pod_agent', log, optional=True, default=True)
            if yp_hard_flag:
                continue

            info['acl'] = get_acl(options, log)
            info['allowed_ssh_key_set'] = _get_item(options, 'allowed_ssh_key_set', log, optional=True, default='unknown')

            for v in options.get("dynamic_attributes", {}).get("labels", {}).get("attributes", []):
                info['labels'][v["key"]] = yson.loads(base64.b64decode(v["value"]))
        except Exception as e:
            log.warning("skipping load infos of pod `%s`: %s", pod, e)
            continue

        pod_id_to_infos_index[pod] = info

    return pod_id_to_infos_index


def _fetch_slots(log=None):
    if log is None:
        log = logging.getLogger('portoshell.fetch-slots')
        log = logging.MessageAdapter(
            log,
            fmt='[%(pid)s] %(message)s',
            data={'pid': os.getpid()},
        )
    try:
        pod_id_to_infos_index = _fetch_yp_lite_pods_infos(log)
    except Exception as e:
        log.warning("skipping load infos of pods: %s", e)
        pod_id_to_infos_index = {}

    for options in http_tools.fetch_json("http://localhost:25536/instances", "ISS", "slots", log=log):
        slot = options.get('slot')
        if slot is None:
            continue

        try:
            config_id = options.get('configurationId')
            if not config_id:
                log.warning("skipping slot with no config_id: `%s`", slot)
                continue

            container = _get_meta_container_path(options, log)
            state = _get_item(options, 'currentState', log)
            target_state = _get_item(options, 'targetState', log)
            instance_dir = _get_item(options, 'instanceDir', log, True)

            properties = options['instanceData']

            nanny_service = _get_item(properties, 'properties/NANNY_SERVICE_ID', log, True)
            nanny_url = _get_item(properties, 'properties/nanny_container_access_url', log, True)
            deploy_engine = _get_item(properties, 'properties/DEPLOY_ENGINE', log, optional=True)
            is_yp_lite = deploy_engine == "YP_LITE"

            pod = None
            pod_labels = None
            yasm_tags = None
            if is_yp_lite:
                pod = slot.split("@")[0]
                pod_labels = pod_id_to_infos_index.get(pod, {}).get('labels')
                yasm_tags = _get_yasm_labels(properties, log)
                acl = pod_id_to_infos_index.get(pod, {}).get('acl', {}).get(box_type_attribute_path('default'))
                allowed_ssh_key_set = pod_id_to_infos_index.get(pod, {}).get('allowed_ssh_key_set')

            project = _get_project(properties, log)
            api_url = _get_item(properties, 'properties/qloudManagementUrl', log, True)

            if not (project and api_url) and not (nanny_service and nanny_url):
                raise Exception("No NANNY_SERVICE_ID nor QLOUD project defined for slot")

            mtn_interfaces = _get_mtn_interfaces(properties, log)
            mtn_hostname = _get_item(properties, 'properties/HOSTNAME', log, True) or None
            mtn_slot_container = _get_mtn_slot_container(options, container, log)
            mtn_ssh_enabled = _get_item(properties, "properties/SKYNET_SSH",
                                        log, True, default='enabled') == 'enabled'
            mtn_host_skynet_enabled = _get_item(properties, "properties/HOST_SKYNET",
                                                log, True, default='disabled') == 'enabled'

            if nanny_url and is_yp_lite:
                yield YpLiteSlot(
                    container=b(container),
                    instance_dir=b(instance_dir),
                    mtn_hostname=mtn_hostname,
                    mtn_interfaces=mtn_interfaces,
                    mtn_slot_container=mtn_slot_container,
                    mtn_ssh_enabled=mtn_ssh_enabled,
                    mtn_host_skynet_enabled=mtn_host_skynet_enabled,
                    api_url=b(nanny_url),
                    service=b(nanny_service),
                    slot=b(slot),
                    configuration_id=b(config_id),
                    state=b(state),
                    target_state=b(target_state),
                    pod=pod,
                    pod_labels=pod_labels,
                    yasm_tags=yasm_tags,
                    acl=acl,
                    allowed_ssh_key_set=allowed_ssh_key_set,
                )
            elif nanny_url and not is_yp_lite:
                yield NannySlot(
                    container=b(container),
                    instance_dir=b(instance_dir),
                    mtn_hostname=mtn_hostname,
                    mtn_interfaces=mtn_interfaces,
                    mtn_slot_container=mtn_slot_container,
                    mtn_ssh_enabled=mtn_ssh_enabled,
                    mtn_host_skynet_enabled=mtn_host_skynet_enabled,
                    api_url=b(nanny_url),
                    service=b(nanny_service),
                    slot=b(slot),
                    configuration_id=b(config_id),
                    state=b(state),
                    target_state=b(target_state)
                )
            else:
                yield QloudSlot(
                    container=b(container),
                    instance_dir=b(instance_dir),
                    mtn_hostname=mtn_hostname,
                    mtn_interfaces=mtn_interfaces,
                    mtn_slot_container=mtn_slot_container,
                    mtn_ssh_enabled=mtn_ssh_enabled,
                    mtn_host_skynet_enabled=mtn_host_skynet_enabled,
                    api_url=b(api_url),
                    project=b(project),
                    slot=b(slot),
                    configuration_id=b(config_id),
                    state=b(state),
                    target_state=b(target_state),
                )
        except Exception as e:
            log.warning("skipping slot `%s`/`%s`: %s", slot, config_id, e)
            continue


@cached(30)
def fetch_slots():
    return {
        k: set(v)
        for k, v in groupby(
            sorted(_fetch_slots(), key=lambda s: s.slot),
            lambda s: s.slot
        )
    }
