import os
import re
import socket
import itertools

import ujson as json

from ya.skynet.util import logging
from ya.skynet.util.functional import cached
from ya.skynet.library.auth.verify import VerifyManager

from infra.skylib.http_tools import fetch_json, fetch_lines

from . import exceptions, auth_mode


class BaseSlot(object):
    __slots__ = [
        'container',
        'instance_dir',
        'mtn_hostname',
        'mtn_interfaces',
        'mtn_slot_container',
        'mtn_ssh_enabled',
        'mtn_host_skynet_enabled',
    ]

    def __init__(self, **kwargs):
        for arg in self.__slots__:
            setattr(self, arg, kwargs.get(arg))

    def __eq__(self, other):
        return type(self) is type(other) and all(
            getattr(self, attr) == getattr(other, attr)
            for attr in self.__slots__
        )

    def __str__(self):
        return '%s(%s)' % (
            type(self).__name__,
            ', '.join('%s=%s' % (attr, getattr(self, attr)) for attr in self.__slots__)
        )

    def __repr__(self):
        return '%s(%s)' % (
            type(self).__name__,
            ', '.join('%s=%r' % (attr, getattr(self, attr)) for attr in self.__slots__)
        )

    def _fqdn(self):
        return 'fqdn=%s' % (self.mtn_hostname or 'unknown')

    def _container(self):
        return 'container=%s' % self.container

    def _additional_info(self):
        return ()

    def as_auth_info(self):
        return '%s(%s)' % (
            type(self).__name__,
            ' '.join(itertools.chain(
                self._additional_info(), (
                    self._fqdn(),
                    self._container(),
                )
            ))
        )

    def get_auth_keys(self, username, keys_storage=None):
        raise exceptions.AuthError("Don't know how to authenticate for %r" % (self,))

    def get_allowed_principals_for(self, username):
        return ()

    def ssh_keys_allowed(self):
        return auth_mode.ssh_keys_allowed()

    def get_allowed_ca_sources(self):
        return auth_mode.get_allowed_cert_kind()

    def get_env_vars(self):
        return ()

    def get_lookup_args(self, strict=True):
        """
        :param bool strict: if set, lookup args must denote one exact slot, that can be disambiuosly
                            matched, otherwise args can match loosely, since exact slot will be
                            augmented by additional info
        """
        return {}

    @classmethod
    def find_best(cls, slot_name, candidates):
        if len(candidates) == 1:
            return next(iter(candidates))

        raise Exception("Cannot select best slot from %d matching candidates" % (len(candidates),))

    def identifier(self):
        raise NotImplementedError


class RawSlot(BaseSlot):
    def get_auth_keys(self, username, keys_storage=None):
        raise exceptions.AuthError("User cannot be authenticated into raw slot")

    def identifier(self):
        return 'CONT:%s' % (self.container,)

    def get_lookup_args(self, strict=True):
        return {'slot_name': self.identifier()}


class QloudSlot(BaseSlot):
    __slots__ = BaseSlot.__slots__ + [
        'api_url',
        'project',
        'slot',
        'configuration_id',
        'state',
        'target_state',
    ]

    def get_auth_keys(self, username, keys_storage=None):
        project_keys = _fetch_qloud_users(self.api_url, self.project)
        if not project_keys:
            raise exceptions.AuthError("No user keys could be found for this slot in registry")
        if username not in project_keys:
            raise exceptions.AuthError("User `%s` is not allowed to join project `%s` containers"
                                       % (username, self.project))

        return project_keys[username]

    def _additional_info(self):
        yield 'slot=%s' % self.slot
        yield 'project=%s' % self.project  # FIXME extract @DOMAIN from api_url?

    @classmethod
    def find_best(cls, slot_name, candidates):
        dest_slot = filter(lambda x: x.state == 'ACTIVE', candidates)
        possible_dest_slot = filter(lambda x: x.target_state == 'ACTIVE', candidates)
        if len(dest_slot) == 1:
            return dest_slot[0]
        elif len(possible_dest_slot) == 1:
            return possible_dest_slot[0]
        elif not dest_slot and not possible_dest_slot and len(candidates) == 1:
            # there is no ACTIVE slots, but we have only one candidate, use it
            return next(iter(candidates))

        raise exceptions.ConfigurationLookupError(
            "Cannot deduce configuration id for slot %r" % (slot_name,),
            [s.configuration_id for s in candidates]
        )

    def identifier(self):
        return '%s / %s' % (self.slot, self.configuration_id)

    def get_lookup_args(self, strict=True):
        if strict:
            return {'slot_name': self.slot, 'configuration_id': self.configuration_id}
        else:
            return {'slot_name': self.slot}


class NannySlot(BaseSlot):
    __slots__ = BaseSlot.__slots__ + [
        'api_url',
        'service',
        'slot',
        'configuration_id',
        'state',
        'target_state',
    ]

    def get_auth_keys(self, username, keys_storage=None):
        user_keys = _fetch_nanny_user_keys(self.api_url, self.service, username)

        if user_keys is False:
            raise exceptions.AuthError("User `%s` is not allowed to join service `%s` containers"
                                       % (username, self.service))
        if not user_keys:
            raise exceptions.AuthError("No user keys could be found for this slot in nanny")

        return user_keys

    def get_env_vars(self):
        srvc_id = re.match(r'^(\w+)@', self.slot)
        if srvc_id:
            srvc_id = srvc_id.group(1)
            yield ('SERVICE_ID', srvc_id)
            try:
                int(srvc_id)
                yield ('SERVICE_PORT', srvc_id)
            except ValueError:
                pass

        yield ('NANNY_SERVICE_ID', self.service)

    def _additional_info(self):
        yield 'slot=%s' % self.slot
        yield 'service=%s' % self.service

    @classmethod
    def find_best(cls, slot_name, candidates):
        dest_slot = filter(lambda x: x.state == 'ACTIVE', candidates)
        possible_dest_slot = filter(lambda x: x.target_state == 'ACTIVE', candidates)
        if len(dest_slot) == 1:
            return dest_slot[0]
        elif len(possible_dest_slot) == 1:
            return possible_dest_slot[0]
        elif not dest_slot and not possible_dest_slot and len(candidates) == 1:
            # there is no ACTIVE slots, but we have only one candidate, use it
            return next(iter(candidates))

        raise exceptions.ConfigurationLookupError(
            "Cannot deduce configuration id for slot %r" % (slot_name,),
            [s.configuration_id for s in candidates]
        )

    def identifier(self):
        return '%s / %s' % (self.slot, self.configuration_id)

    def get_lookup_args(self, strict=True):
        if strict:
            return {'slot_name': self.slot, 'configuration_id': self.configuration_id}
        else:
            return {'slot_name': self.slot}

    @property
    def slot_name(self):
        return self.slot


class YpLiteSlot(NannySlot):
    __slots__ = NannySlot.__slots__ + [
        'pod',
        'pod_labels',
        'yasm_tags',
        'acl',
        'allowed_ssh_key_set',
    ]

    def ssh_keys_allowed(self):
        label = self.pod_labels.get("allowed_ssh_key_set")
        return auth_mode.ssh_keys_allowed(label, self.allowed_ssh_key_set)

    def get_allowed_ca_sources(self):
        label = self.pod_labels.get("allowed_ssh_key_set")
        return auth_mode.get_allowed_cert_kind(label, self.allowed_ssh_key_set)

    def get_auth_keys(self, username, keys_storage=None):
        if not self.ssh_keys_allowed():
            user_keys = []
        else:
            try:
                user_keys = _fetch_nanny_user_keys(
                    self.api_url,
                    self.service,
                    username,
                    pod=self.pod,
                    acl=self.acl,
                    keys_storage=keys_storage,
                )
            except exceptions.AuthError:
                if self.pod and self.acl and keys_storage:
                    # FIXME right now nanny may delay acl export or export not all groups, so
                    # in case of keys or acl missing, retry without acl.
                    # Shoult be removed after DEPLOY-5016.
                    user_keys = _fetch_nanny_user_keys(
                        self.api_url,
                        self.service,
                        username,
                    )
                else:
                    raise

        if user_keys is False:
            raise exceptions.AuthError("User `%s` is not allowed to join service `%s` containers"
                                       % (username, self.service))
        if not user_keys:
            raise exceptions.AuthError("No user keys could be found for this slot in nanny")

        return user_keys

    def get_allowed_principals_for(self, username):
        if self.acl and self.pod:
            allowed_users = _fetch_yp_allowed_users(
                pod=self.pod,
                paths=[box_type_attribute_path('default')],
                username=username,
                acl=self.acl,
                yp_lite_mode=True,
            )

            try:
                return _yp_make_lookup_user_list(username, allowed_users, self.pod)
            except exceptions.AuthError:
                pass

        return ()


class CocaineSlot(BaseSlot):
    __slots__ = BaseSlot.__slots__ + ['acl', 'app', 'uuid']

    def get_auth_keys(self, username, keys_storage=None):
        if username not in self.acl:
            raise exceptions.AuthError("User `%s` is not allowed to use cocaine container `%s` (denied by ACL: `%s`)"
                                       % (username, self.container, self.acl))
        # probably we could use keys_storage here, but there's no ISS on cocaine hosts
        return _fetch_ldap_user_keys(username=username)

    def _additional_info(self):
        yield 'app=%s' % self.app
        yield 'uuid=%s' % self.uuid

    def identifier(self):
        return 'app=%s' % (self.app)

    def get_lookup_args(self, strict=True):
        return {'slot_name': 'cocaine/%s' % (self.uuid,)}


class YpSlot(BaseSlot):
    BOX_ID_FQDN_RE = re.compile("[a-zA-Z0-9-]{1,63}")

    __slots__ = BaseSlot.__slots__ + ['pod', 'box', 'api_url', 'pod_set_id', 'box_type', 'acl',
                                      'pod_labels', 'allowed_ssh_key_set']

    def ssh_keys_allowed(self):
        label = self.pod_labels.get("allowed_ssh_key_set")
        return auth_mode.ssh_keys_allowed(label, self.allowed_ssh_key_set)

    def get_allowed_ca_sources(self):
        label = self.pod_labels.get("allowed_ssh_key_set")
        return auth_mode.get_allowed_cert_kind(label, self.allowed_ssh_key_set)

    def get_auth_keys(self, username, keys_storage=None):
        if not self.ssh_keys_allowed():
            return []

        return _fetch_yp_user_keys(self.pod, self.box_type, username, self.acl, keys_storage=keys_storage)

    def get_allowed_principals_for(self, username):
        allowed_users = _fetch_yp_allowed_users(
            pod=self.pod,
            paths=[None, box_type_attribute_path(self.box_type)],
            username=username,
            acl=self.acl,
        )

        try:
            return _yp_make_lookup_user_list(username, allowed_users, self.pod)
        except exceptions.AuthError:
            return ()

    def get_env_vars(self):
        yield ('YP_POD', self.pod)
        if self.box:
            yield ('YP_BOX', self.box)

    def _additional_info(self):
        yield 'pod=%s' % (self.pod,)
        if self.box:
            yield 'box=%s' % (self.box,)
        yield 'pod_set_id=%s' % (self.pod_set_id,)

    def identifier(self):
        return '%s/%s' % (self.pod, self.box) if self.box else self.pod

    def get_lookup_args(self, strict=True):
        return {'pod': self.pod, 'box': self.box}

    def get_persistent_pod_fqdn(self):
        """
        mtn_hostname is transient pod fqdn (example: {node-id}.{pod-id}.sas-test.yp-test.yandex.net),
        but in future maybe changed to persistent (example: {pod-id}.sas-test.yp-test.yandex.net)

        :rtype: str | None
        """
        if not self.mtn_hostname:
            return None
        parts = []
        found_pod_id = False
        for part in self.mtn_hostname.split("."):
            if part == self.pod:
                found_pod_id = True
            if found_pod_id:
                parts.append(part)
        if found_pod_id:
            return ".".join(parts)

    def get_persistent_box_fqdn(self):
        """

        https://st.yandex-team.ru/YP-2654

        """
        if not self.box:
            return None

        if not self.BOX_ID_FQDN_RE.match(self.box):
            return None

        persistent_pod_fqdn = self.get_persistent_pod_fqdn()
        if not persistent_pod_fqdn:
            return None

        return "{}.{}".format(self.box.lower(), persistent_pod_fqdn)


@cached(300)
def _fetch_qloud_users(api_url, project, username):
    log = logging.getLogger('portoshell.fetch-users')
    log = logging.MessageAdapter(
        log,
        fmt='[%(pid)s] %(message)s',
        data={'pid': os.getpid()},
    )

    url = "%s/api/v1/ssh-users/%s" % (api_url, project)
    data = fetch_json(url, 'Qloud', 'users', log=log)

    result = {}

    for user_info in data:
        try:
            user = user_info["login"]
            sshkeys = user_info["sshKeys"]
            keys = []
            for key in sshkeys:
                try:
                    for k in VerifyManager.loadsKeys(key["text"], key["description"]):
                        if 'cert-authority' in k.options:
                            continue
                        k.userNames.add(user)
                        keys.append(k)
                except Exception as e:
                    log.warning('skipping key `%s`: %s', key['description'], e)

            result[user] = keys

        except Exception as e:
            log.warning("cannot fetch keys: %s", e)
            continue

    return result


@cached(300)
def _fetch_nanny_user_keys(nanny_url, service_id, username, pod=None, acl=None, keys_storage=None):
    log = logging.getLogger('portoshell.fetch-nanny')
    log = logging.MessageAdapter(
        log,
        fmt='[%(pid)s] %(message)s',
        data={'pid': os.getpid()},
    )

    if acl and pod and keys_storage:
        allowed_users = _fetch_yp_allowed_users(
            pod=pod,
            paths=[box_type_attribute_path('default')],
            username=username,
            acl=acl,
            yp_lite_mode=True,
        )

        users = _yp_make_lookup_user_list(username, allowed_users, pod)

        try:
            return set(itertools.chain.from_iterable(
                keys_storage.get_keys(user)
                for user in users
            ))
        except keys_storage.KeysNotReady:
            pass

    data = {
        'login': username,
        'serviceId': service_id,
    }
    headers = {
        'Content-Type': 'application/json',
    }
    data = fetch_json(nanny_url, 'Nanny', 'user keys', data=json.dumps(data), headers=headers, log=log)

    if not data['accessGranted']:
        return False

    keys = []
    for k in data['keys']:
        try:
            key = VerifyManager.loadsKeys(k['key']).next()
        except Exception as e:
            log.warning("cannot parse key: %s", e)
        else:
            if 'cert-authority' in key.options:
                continue
            key.userNames.add(username)
            keys.append(key)

    return keys


@cached(300)
def _fetch_ldap_user_keys(username):
    log = logging.getLogger('portoshell.fetch-ldap')
    log = logging.MessageAdapter(
        log,
        fmt='[%(pid)s] %(message)s',
        data={'pid': os.getpid()},
    )

    urls = tuple('%s/%s/' % (ldap_url, username)
                 for ldap_url in ("https://ldap-dev.yandex.net:4443/userkeys",
                                  "http://ldap-dev.yandex.net:4444/userkeys")
                 )

    lines = []
    for i in fetch_lines(urls, 'LDAP', 'user keys', log=log):
        if i[0] == '#':
            continue
        else:
            lines.append(i)

    if not lines:
        return False

    keys = []
    for k in lines:
        try:
            key = VerifyManager.loadsKeys(k).next()
        except Exception as e:
            log.warning("cannot parse key: %s", e)
        else:
            if 'cert-authority' in key.options:
                continue
            key.userNames.add(username)
            keys.append(key)

    return keys


def _fetch_yp_allowed_users(pod, paths, username, acl, yp_lite_mode=False):
    # Deploy:
    # "root_ssh_access" allows to authorize as "root" or "nobody"
    # "ssh_access" allows to authorize as "nobody"
    # "ssh_access" and <user> keys allow to authorize as <user>

    # YpLite:
    # in Nanny <user> means root, thus we have:
    # "root_ssh_access" allows to authorize as "root", "nobody" and <user>
    # "ssh_access" allows to authorize as "nobody"

    # if acl is present, we retrieve allowed users from there,
    # otherwise we fallback to yp requests

    allowed_users = set()

    if acl is not None:
        if username in ('root', 'nobody') or yp_lite_mode:
            allowed_users.update(acl.get('root_ssh_access', []))
        if username != 'root' and not yp_lite_mode:
            allowed_users.update(acl.get('ssh_access', []))
        elif username == 'nobody' and yp_lite_mode:
            allowed_users.update(acl.get('ssh_access', []))
        return list(allowed_users)

    return list(allowed_users)


def box_type_attribute_path(box_type):
    return '/access/deploy/box/%s/' % box_type


def _yp_make_lookup_user_list(username, allowed_users, pod):
    if (username not in ('root', 'nobody') and username not in allowed_users) or not allowed_users:
        raise exceptions.AuthError("User %r is not allowed to join pod %r containers"
                                   % (username, pod))

    if username in ('root', 'nobody'):
        return [user for user in allowed_users if user not in ('nobody', 'root')]
    else:
        return [username]


@cached(300)
def _fetch_yp_user_keys(pod, box_type, username, acl, keys_storage=None):
    log = logging.getLogger('portoshell.fetch-pods')
    log = logging.MessageAdapter(
        log,
        fmt='[%(pid)s] %(message)s',
        data={'pid': os.getpid()},
    )

    paths = [None, box_type_attribute_path(box_type)]

    allowed_users = _fetch_yp_allowed_users(pod, paths, username, acl)

    users = _yp_make_lookup_user_list(username, allowed_users, pod)

    if keys_storage is not None:
        try:
            return set(itertools.chain.from_iterable(
                keys_storage.get_keys(user)
                for user in users
            ))
        except keys_storage.KeysNotReady:
            pass

    return []


def normalize_ip(address, family=None):
    if family is None:
        candidates = (socket.AF_INET6, socket.AF_INET)
    else:
        candidates = (family,)

    for candidate in candidates:
        try:
            return socket.inet_ntop(candidate, socket.inet_pton(candidate, address))
        except Exception:
            continue

    return address
