import os
import re
import sys
import socket

from porto.api import Connection
from ya.skynet.util import logging

from infra.skylib.http_tools import fetch_json

from . import slot, exceptions, iss_fetcher, yp_fetcher


RAW_CONTAINER_PREFIX = 'CONT:'
SERVICE_PREFIX_COCAINE = 'cocaine/'

if sys.version_info.major < 3:
    def b(s):
        if s is None:
            return None
        return s.encode('utf-8') if isinstance(s, unicode) else str(s)  # noqa
else:
    def b(s):
        if s is None:
            return None
        return s if isinstance(s, bytes) else bytes(str(s), 'utf-8')


class RawSource(object):
    @classmethod
    def filter(cls, **kwargs):
        return 'slot_name' in kwargs and kwargs['slot_name'].startswith(RAW_CONTAINER_PREFIX)

    def find_slots(self, **kwargs):
        slot_name = kwargs['slot_name']
        yield slot.RawSlot(
            slot=b(slot_name),
            container=b(slot_name[5:]),
            instance_dir='/',
        )


class CocaineSource(object):
    @classmethod
    def filter(cls, **kwargs):
        if 'slot_name' not in kwargs:
            return False
        slot_name = kwargs['slot_name']
        return (
            slot_name.startswith(SERVICE_PREFIX_COCAINE)
            or slot_name.find(':' + SERVICE_PREFIX_COCAINE) >= 0
        )

    def _cocaine_resolve_app(self, slot_name):
        log = logging.getLogger('portoshell.cocaine-resolve-app')
        log = logging.MessageAdapter(
            log,
            fmt='[%(pid)s] %(message)s',
            data={'pid': os.getpid()},
        )
        fqdn = socket.gethostname()
        starts_at = slot_name.find(SERVICE_PREFIX_COCAINE)
        uuid = slot_name[starts_at + len(SERVICE_PREFIX_COCAINE):]
        portoconn = Connection()
        portoconn.connect()
        for c in portoconn.ListContainers():
            cmd = c.GetProperty('command')
            s = re.search(r'--app\s+(\S+)\b', cmd)
            app = s.group(1) if s else None
            s = re.search(r'--uuid\s+(\S+)\b', cmd)
            current_worker_uuid = s.group(1) if s else None

            cwd = c.GetProperty('cwd')
            if app and current_worker_uuid and cwd:
                if uuid == current_worker_uuid:
                    acl = self._fetch_cocaine_acl(app=app, fqdn=fqdn)
                    if acl is not None:
                        return slot.CocaineSlot(
                            container=b(c.name),
                            instance_dir=b(cwd),
                            acl=b(acl),
                            app=b(app),
                        )
                    else:
                        raise Exception("No ACL found for app='%s' with worker uuid='%s'." % (app, current_worker_uuid))
        raise exceptions.SlotLookupError(message='Cocaine worker not found.', slots=[])

    def _fetch_cocaine_acl(self, app, fqdn):
        log = logging.getLogger('portoshell.fetch-cocaine-acl')
        log = logging.MessageAdapter(
            log,
            fmt='[%(pid)s] %(message)s',
            data={'pid': os.getpid()},
        )

        url = 'https://pipeline.ape.yandex-team.ru/acl4app/%s/%s' % (app, fqdn)
        data = fetch_json(url, 'cocaine', 'ACL', log=log)

        acl = []

        # /secure/acl/<appname>/<UID>/shell
        try:
            for uid in data['secure']['acl'][app].keys():
                try:
                    if data['secure']['acl'][app][uid]['shell'] == 'allow':
                        acl.append(uid)
                        log.debug('added uid="%s" to ACL', uid)
                except KeyError:
                    pass
        except KeyError:
            raise Exception('Malformed ACL for application %s', app)

        return tuple(acl) or None

    def find_slots(self, **kwargs):
        yield self._cocaine_resolve_app(kwargs['slot_name'])


class IssSource(object):
    @classmethod
    def filter(cls, **kwargs):
        if 'slot_name' not in kwargs or not kwargs.get('iss_enabled', True):
            return False

        if RawSource.filter(**kwargs) or CocaineSource.filter(**kwargs):
            return False

        return True

    def find_slots(self, **kwargs):
        slot_name = kwargs['slot_name']
        configuration_id = kwargs.get('configuration_id')

        try:
            slots = iss_fetcher.fetch_slots()
        except Exception as e:
            raise Exception("failed to fetch available slots: %s" % (e,))

        if slots is None or slot_name not in slots:
            raise exceptions.SlotLookupError("No slot `%s` on this host" % slot_name, slots.items() if slots else [])

        candidates = slots[slot_name]
        if configuration_id:
            subcandidates = filter(lambda x: x.configuration_id == configuration_id, candidates)
            if not subcandidates:
                raise exceptions.ConfigurationLookupError(
                    "Cannot find configuration id %r for slot %r" % (configuration_id, slot_name),
                    [s.configuration_id for s in candidates]
                )
            return subcandidates

        return candidates


class YpSource(object):
    @classmethod
    def filter(cls, **kwargs):
        if 'pod' not in kwargs or not kwargs.get('yp_enabled', True):
            return False

        if RawSource.filter(**kwargs) or CocaineSource.filter(**kwargs):
            return False

        if not yp_fetcher.get_yp_host():
            return False

        return True

    def find_slots(self, **kwargs):
        pod_name = kwargs['pod']
        box = kwargs.get('box')

        try:
            pods = yp_fetcher.fetch_slots()
        except Exception as e:
            raise Exception("failed to fetch available slots: %s" % (e,))

        matching_pods = filter(lambda slot: slot.pod == pod_name, pods)
        if not matching_pods:
            raise exceptions.SlotLookupError("No pod `%s` on this host" % pod_name, pods)

        if box:
            matching_boxes = filter(lambda slot: slot.box == box, matching_pods)
        else:
            matching_boxes = filter(lambda slot: slot.box is None, matching_pods)

        if not matching_boxes:
            raise exceptions.ConfigurationLookupError(
                "Cannot find box %r for pod %r" % (box, pod_name),
                [slot.box for slot in matching_pods]
            )
        return matching_boxes


fetchers = [
    RawSource(),
    CocaineSource(),
    IssSource(),
    YpSource(),
]


def find_best_matching_slot(slot_name, candidates, permissive=False):
    if len(candidates) == 1:
        return next(iter(candidates))

    first_type = type(next(iter(candidates)))

    if not all(type(c) is first_type for c in candidates):
        return
    else:
        try:
            return first_type.find_best(slot_name, candidates)
        except Exception:
            if not permissive:
                raise


def find_slot(**kwargs):
    log = logging.getLogger('portoshell.find_slot')
    # FIXME need some other way to find and raise "slot" name
    slot_name = kwargs.get('slot_name', kwargs.get('pod', ''))
    result = None
    exc = None
    for fetcher in fetchers:
        if not fetcher.filter(**kwargs):
            log.debug("fetcher %s not suitable for %s", type(fetcher).__name__, kwargs)
            continue
        try:
            result = list(fetcher.find_slots(**kwargs))
        except Exception as e:
            # TODO merge slot lookup errors from Nanny and YP
            log.debug("lookup in %s failed with %s", type(fetcher).__name__, e)
            exc = e

        if result:
            log.debug("got %s, will select best", result)
            return find_best_matching_slot(slot_name, result)

    if exc:
        raise exc

    raise exceptions.SlotLookupError("No slot %r on this host" % slot_name, [])
