import os
import sys
import base64

from yt import yson

from .slot import YpSlot, box_type_attribute_path, normalize_ip

from infra.skylib import http_tools

from ya.skynet.util import logging
from ya.skynet.util.functional import cached


__all__ = ['fetch_slots', 'get_yp_host']


if sys.version_info.major < 3:
    def b(s):
        if s is None:
            return None
        return s.encode('utf-8') if isinstance(s, unicode) else str(s)  # noqa
else:
    def b(s):
        if s is None:
            return None
        return s if isinstance(s, bytes) else bytes(str(s), 'utf-8')


def _get_container_path(options, log):
    container = options.get('container_name')
    log.debug('read meta container path: %s', container)
    return container


def _get_item(dictionary, item, log, optional=False, default=None):
    try:
        result = dictionary.get(item, default) if optional else dictionary[item]
        log.debug('read %s: %s', item, result)
        return result
    except KeyError:
        log.debug('got no %s', item)
        raise


def _get_mtn_interfaces(properties, log):
    mtn_interfaces = []
    for allocation in properties.get('ip6_address_allocations', []):
        # TODO also add subnets
        address = allocation.get('address')
        if address:
            mtn_interfaces.append(b(normalize_ip(address)))

    if not mtn_interfaces:
        mtn_interfaces = None
        log.debug('got no mtn interfaces')

    return mtn_interfaces


def get_acl(properties, log):
    enum_to_acl_name = {
        'ACA_SSH_ACCESS': 'ssh_access',
        'ACA_ROOT_SSH_ACCESS': 'root_ssh_access',
    }

    acl = {}
    acl_objects = properties.get('access_permissions')
    if not acl_objects:
        return None

    for acl_object in acl_objects:
        attribute_path = acl_object.get('attribute_path')
        permission = enum_to_acl_name.get(acl_object.get('permission'))
        user_ids = acl_object.get('user_ids')
        if attribute_path is None or permission is None or not isinstance(user_ids, list):
            continue

        if not attribute_path.endswith('/'):
            attribute_path += '/'

        for user_id in user_ids:
            acl.setdefault(attribute_path, {}).setdefault(permission, set()).add(b(user_id))

    log.debug('read acl: %s', acl)
    return acl


def _fetch_slots(log=None):
    if log is None:
        log = logging.getLogger('portoshell.fetch-pods')
        log = logging.MessageAdapter(
            log,
            fmt='[%(pid)s] %(message)s',
            data={'pid': os.getpid()},
        )

    for options in http_tools.fetch_json("http://localhost:25536/pods/info", "YP", "pods", log=log).get("pod_info", []):
        pod = options.get('id')
        if pod is None:
            continue

        try:
            yp_hard_flag = _get_item(options, 'has_pod_agent', log, optional=True, default=True)
            if not yp_hard_flag:
                continue

            container = _get_container_path(options, log)
            fqdn = _get_item(options, 'host_name', log)
            box = None
            box_type = 'system'

            # instance_dir = _get_item(options, 'instanceDir', log, True)
            instance_dir = '/'  # FIXME

            api_url = '%s:8090' % (get_yp_host(log=log),)

            mtn_interfaces = _get_mtn_interfaces(options, log)
            mtn_ssh_enabled = True  # FIXME
            mtn_host_skynet_enabled = True  # FIXME (torkve) we should in fact check if skynet is bound to box
            if not mtn_interfaces:
                continue

            acl = get_acl(options, log)
            pod_labels = {}
            for v in options.get("dynamic_attributes", {}).get("labels", {}).get("attributes", []):
                pod_labels[v["key"]] = yson.loads(base64.b64decode(v["value"]))

            pod_set_id = _get_item(options, 'pod_set_id', log, optional=True, default='<unknown>')

            # This value affects ONLY 'default' box_types. Pod and 'system' boxes are forced to use 'secure'
            allowed_ssh_key_set = _get_item(options, 'allowed_ssh_key_set', log, optional=True, default='unknown')

            yield YpSlot(
                container=b(container),
                instance_dir=b(instance_dir),
                mtn_hostname=fqdn,
                mtn_interfaces=mtn_interfaces,
                mtn_slot_container=container,
                mtn_ssh_enabled=mtn_ssh_enabled,
                api_url=b(api_url),
                pod=b(pod),
                box=box,
                box_type=b(box_type),
                acl=acl.get(box_type_attribute_path(box_type), {}) if acl is not None else None,
                allowed_ssh_key_set='secure',
                mtn_host_skynet_enabled=mtn_host_skynet_enabled,
                pod_labels=pod_labels,
                pod_set_id=pod_set_id,
            )

            boxes = _get_item(options, 'box_info', log, optional=True, default=[])
            for box_info in boxes:
                ip_address = _get_item(box_info, 'ip6_address', log, optional=True)
                if ip_address:
                    box_id = _get_item(box_info, 'id', log)
                    box_type = _get_item(box_info, 'box_type', log, optional=True, default=None) or 'default'
                    yield YpSlot(
                        container=b(_get_container_path(box_info, log)),
                        instance_dir=b(instance_dir),  # FIXME too
                        mtn_hostname='%s.%s' % (box_id, fqdn),
                        mtn_interfaces=[b(ip_address)],
                        mtn_slot_container=container,
                        mtn_ssh_enabled=mtn_ssh_enabled,
                        api_url=b(api_url),
                        pod=b(pod),
                        box=b(box_id),
                        box_type=b(box_type),
                        acl=acl.get(box_type_attribute_path(box_type), {}) if acl is not None else None,
                        allowed_ssh_key_set='secure' if box_type == 'system' else allowed_ssh_key_set,
                        mtn_host_skynet_enabled=mtn_host_skynet_enabled,
                        pod_labels=pod_labels,
                        pod_set_id=pod_set_id,
                    )

        except Exception as e:
            log.warning("skipping slot %r/%r: %s", pod, box, e)
            continue


@cached(600)
def get_yp_host(log=None):
    if log is None:
        log = logging.getLogger('portoshell.fetch-pods')
        log = logging.MessageAdapter(
            log,
            fmt='[%(pid)s] %(message)s',
            data={'pid': os.getpid()},
        )

    try:
        data = http_tools.fetch_json("http://localhost:25536/config", "YP", "master host", log=log)
        host = data.get("agent", {}).get("hostConfiguration", {}).get("ypProvider", {}).get("host")
        if not host:
            log.debug("no YP provider available")
        else:
            return host
    except Exception as e:
        log.warning("Could not find yp master host: %s", e)


@cached(30)
def fetch_slots():
    return list(_fetch_slots())
