import copy
import sys
import re
import os
import py
import yaml
import stat

import subprocess as sp
from contextlib import contextmanager

from .kernel_util import size2str

from .utils import auto_user_privileges, getuid, getgid


@contextmanager
def dummy(*args, **kwargs):
    yield


def get_coredump_meta_path(fp):
    return fp.dirpath('{0}.meta'.format(fp.basename))


def remove_coredump_meta(log, fp):
    meta = get_coredump_meta_path(fp)
    if meta.check(file=1):
        try:
            meta.remove()
        except py.error.Error as ex:
            log.warning(str(ex))


class NamingPatternParseException(Exception):
    def __init__(self, field):
        super(NamingPatternParseException, self).__init__()
        self.field = field


class Entity(object):
    """
    The class is designed as simple slotted data entry storage with an ability to fill in
    slots available via keyword arguments in its constructor. In case of no value provided
    to the constructor, it will be fetched out from `__defs__` array (which should be of
    the same length as `__slots__`).
    """
    __slots__ = []
    __defs__ = []

    def __init__(self, *args, **kwargs):
        for i, attr in enumerate(self.__slots__):
            _def = self.__defs__[i]
            if i < len(args):
                val = args[i]
            elif attr in kwargs:
                val = kwargs[attr]
            else:
                val = copy.copy(_def)
                _def = None  # Avoid double checking
            setattr(self, attr, val if _def is None or isinstance(val, _def.__class__) else _def.__class__(val))

    def __repr__(self):
        return self.__class__.__name__ + repr(dict(self))

    def __iter__(self):
        for attr in self.__slots__:
            yield (attr, getattr(self, attr))

    def itervalues(self):
        for attr in self.__slots__:
            yield getattr(self, attr)

    def copy(self):
        return self.__class__([v for _, v in self])


class CoreCollectorConfig(Entity):
    """
    Configures Procman's core dumps collector. Follows properties' meaning:
        folder: path to a folder the cores should be collected in. Defaults to "<PROCMAN_WORKDIR>/coredumps"
        policy: one of the following strings:
            'copy': copy original file into the folder specified.
            'move': move the original file.
            'symlink': Default. make a symlink to the original file.
        amount: limits the amount of core files to be collected.
        size: limits the size summary of collected core files.
    """
    __slots__ = ['folder', 'user_folder', 'policy', 'amount', 'size']
    __defs__ = [None, False, 'symlink', sys.maxint, sys.maxint]

    @staticmethod
    def linux_replaces(pid):
        """
        Linux core file naming template-to-regular expression replaces.
        """
        return {
            '%': '\%',             # Escaping
            'p': str(pid),         # PID of dumped process
            'u': '\d+?',           # real UID of dumped process
            'g': '\d+?',           # real GID of dumped process
            's': '\d+?',           # number of signal causing dump
            't': '\d+?',           # time of dump, expressed as seconds since the Epoch
            'h': '[0-9a-z_.-]+?',  # hostname (same as nodename returned by uname(2))
            'e': '.+?',            # executable filename (without path prefix)
            'E': '.+?',            # pathname of executable, with slashes ('/') replaced by exclamation marks ('!')
            'c': '\d+?'            # core file size soft resource limit of crashing process
        }

    @staticmethod
    def bsd_replaces(pid):
        """
        *BSD core file naming template-to-regular expression replaces.
        """
        return {
            '%': '\%',             # Escaping
            'H': '[0-9a-z_.-]+?',  # Hostname
            'I': '\d+?',           # Core index
            'N': '.*?',            # Process name
            'P': str(pid),         # Process ID
            'U': '\d+?'            # Process UID
        }

    @staticmethod
    def naming_pattern(template, replaces):
        """
        Converts core file name template into a regular expression.
        """
        r = re.compile(r'([^%])%([a-z%])', re.IGNORECASE)
        pattern = ''
        while True:
            m = r.search(template)
            if not m:
                break
            if m.group(2) not in replaces:
                # we cannot parse this, may be we should just assume '.*?' ?
                raise NamingPatternParseException(m.group(2))

            pattern += re.escape(template[:m.start()] + m.group(1)) + replaces[m.group(2)]
            template = template[m.end():]
        return re.compile('^' + pattern + re.escape(template) + '$', re.IGNORECASE)

    def to_dict(self):
        ret = dict(self)
        ret['folder'] = self.folder.strpath
        return ret

    def ensure_folder(self, uid):
        if self.user_folder and uid:
            with auto_user_privileges(uid):
                self.folder.ensure(dir=1)

    def collect(self, ctx, log, cwd, uid, pid, uuid, stoptime):
        self.ensure_folder(uid)

        uname = os.uname()[0].lower()
        if uname == 'freebsd':
            tmpl = sp.check_output(['/sbin/sysctl', '-n', 'kern.corefile']).strip()
            replaces = CoreCollectorConfig.bsd_replaces(pid)
        elif uname == 'darwin':
            tmpl = sp.check_output(['/usr/sbin/sysctl', '-n', 'kern.corefile']).strip()
            replaces = CoreCollectorConfig.bsd_replaces(pid)
        elif uname == 'linux':
            tmpl = py.path.local('/proc/sys/kernel/core_pattern').read().strip()
            replaces = CoreCollectorConfig.linux_replaces(pid)
            if py.path.local('/proc/sys/kernel/core_uses_pid').read().strip() != '0':
                tmpl += '.%p'
        else:
            log.warning('Core dump cannot be collected - dont know how to do this on %s platform', uname)
            return

        if '|' in tmpl:
            log.warning('Core dump cannot be collected - is piped to executable: %r', tmpl)
            return

        # if some templates are used in dirname, we will never traverse all the directory structure hoping for luck
        if os.path.sep in tmpl and '%' in tmpl.rsplit(os.path.sep, 1)[0].replace('%%', ''):
            log.warning('Core dump cannot be collected - templates are used in dirname: %r', tmpl)
            return

        try:
            regex = CoreCollectorConfig.naming_pattern(tmpl, replaces)
        except NamingPatternParseException as e:
            log.warning("Core dump cannot be collected with certainty - don't know how to parse the %s template, will use heuristics", e.field)
            regex = re.compile(r'^$')  # proven to never match filename
        except Exception as e:
            log.warning("Core dump cannot be collected - error occurred: %s", e)
            return

        path = py.path.local(cwd if tmpl[0] != os.path.sep else os.path.sep)
        for d in tmpl.split(os.path.sep):
            newpath = path.join(d)
            if not newpath.check(dir=1):
                break
            path = newpath

        log.debug('Checking directory %r for core files matching pattern %r', path.strpath, regex.pattern)
        corepath = None
        core_heu_candidates = []
        for fpath in path.listdir():
            fstat = fpath.stat()
            # log.debug("checking candidate %r with uid %s (required %s) and mtime %s (required stoptime %s)", fpath.strpath, fstat.uid, uid, fstat.mtime, stoptime)
            if regex.match(fpath.basename) or regex.match(fpath.strpath):
                corepath = fpath
                break
            # we intentionally set small threshold of 3 seconds, because some guys run processes who
            # segfault more than work. If delay between segfault and collect is greater... well, you
            # are unlucky
            elif fstat.uid == uid and (stoptime - 3) < fstat.mtime <= stoptime:
                core_heu_candidates.append((stoptime - fstat.mtime, fpath))
        else:
            core_heu_candidates.sort()
            if core_heu_candidates:
                corepath = core_heu_candidates[0][1]
                log.info("No matching core dump file found at %r by pattern %r (naming template is %r), heuristics found file %r",
                         path.strpath, regex.pattern, tmpl, corepath.strpath,
                         )
            else:
                log.warning(
                    'No matching core dump file found at %r by pattern %r (naming template is %r)',
                    path.strpath, regex.pattern, tmpl
                )

                return

        storage = self.folder
        target = storage.join('%s.%s' % (uuid, pid))
        ouid = os.getuid()

        if uid:
            class Log(object):
                def __init__(self, slave):
                    self.slave = slave

                def fw(self, meth, *args, **kwargs):
                    with auto_user_privileges(ouid):
                        return getattr(self.slave, meth)(*args, **kwargs)

                def debug(self, *args, **kwargs):
                    return self.fw('debug', *args, **kwargs)

                def info(self, *args, **kwargs):
                    return self.fw('info', *args, **kwargs)

                def warning(self, *args, **kwargs):
                    return self.fw('warning', *args, **kwargs)

            log = Log(log)

        with auto_user_privileges(uid):
            if self.policy == 'copy':
                corepath.copy(target)
            elif self.policy == 'move':
                corepath.move(target)
            else:
                target.mksymlinkto(corepath)

            meta = get_coredump_meta_path(target)
            metainfo = {
                'uuid': uuid,
                'env': ctx['env'],
                'cwd': cwd,
                'uid': uid,
                'pid': pid,
                'args': ctx['args'],
            }

            yaml.dump(metainfo, meta.open('w'), default_flow_style=False)

            # Now check the folder for limits. But first, remove any broken symlinks.
            # List all the files (which ends with '.PID').
            extre = re.compile(r'^\.\d+$')
            files = [(fp, fp.realpath()) for fp in storage.listdir() if extre.match(fp.ext)]
            for fp, rfp in files:
                if fp.check(link=1) and not rfp.check(exists=1):
                    log.warning('Dropping broken link %r', fp.strpath)

                    remove_coredump_meta(log, fp)

                    try:
                        fp.remove()
                    except py.error.Error as ex:
                        log.warning(str(ex))

            # Now filter out any junk here.
            files = sorted(
                ((fp, rfp) for fp, rfp in files if (fp.check(link=1) or fp.check(file=1)) and rfp.check(file=1)),
                key=lambda x: x[1].stat().mtime,
                reverse=True
            )

            # Collect the storage utilization counters.
            size = sum(rfp.stat().size for _, rfp in files)
            amount = len(files)

            log.debug(
                'New core dump stored as %r. Storage utilization: %d/%s files of %s/%s size totally.',
                target.strpath,
                amount, 'unlimited' if self.amount == sys.maxint else str(self.amount),
                size2str(size), 'unlimited' if self.size == sys.maxint else size2str(self.size)
            )

            # Calculate files list to remove (outdated).
            remove = files[self.amount:amount] if amount > self.amount else []
            size -= sum(rfp.lstat().size for _, rfp in remove)
            while size > self.size:
                remove.append(files.pop())
                size -= remove[-1][1].stat().size

            if not remove:
                return target.strpath

            seen = set()
            for fp, rfp in remove:
                if fp in seen:
                    continue
                log.info('Dropping outdated core file: %s', fp)
                seen.add(fp)
                try:
                    rfp.remove()
                except py.error.Error as ex:
                    log.warning(str(ex))

                remove_coredump_meta(log, fp)

                if fp.check(link=1):
                    try:
                        fp.remove()
                    except py.error.Error as ex:
                        log.warning(str(ex))

        return target.strpath
