from gevent import socket, spawn, Timeout, GreenletExit, sleep, event
from functools import partial

import os
import grp
import pwd
import pty
import sys
import time
import struct
import errno
import signal
import logging
import msgpack
import subprocess
import traceback
import resource

import six

try:
    from porto.exceptions import ContainerDoesNotExist, InvalidProperty, UnknownError, NotSupported, EError
except ImportError:
    pass

from . import errors, portoutils
from .utils import dummy, has_root, Fd, liner, build_limits, gread, auto_user_privileges, geteuid, open_file

from .kernel_util.errors import formatException
from .kernel_util.functional import singleton
from .kernel_util.misc import daemonthr
from .kernel_util.sys.user import UserPrivileges, getUserResourceLimits, setResourceLimits
from .kernel_util.sys.getpeerid import getpeerid
from .mailbox import mailbox

msgpack_exceptions = (msgpack.UnpackException,) if msgpack.version > (0, 3) else ()
msgpack_exceptions += (OverflowError, ValueError, TypeError, StopIteration)


def runproc(
    log, args, env, cwd, uid=None, peer_uid=None, cpus=None,
    pty=True, limits=None, cgroup_inherited=None, cgroup=None
):
    if limits is None:
        limits = {}
    user_limits = getUserResourceLimits(uid, defaults={}) if uid is not None else None

    def prefunc():
        for handler in logging.getLogger('').handlers:
            if hasattr(handler, 'stream'):
                if handler.__class__.__name__ == 'StreamHandler':
                    logging.getLogger('').removeHandler(handler)
                else:
                    handler.stream = None

        try:
            os.chdir(cwd)
            if cpus:
                # Nothing to do with return code..
                os.system('cpuset -Cl %s -p %d' % (','.join(map(str, cpus)), os.getpid()))
        except EnvironmentError:
            pass

        cgroups_root = '/sys/fs/cgroup'
        pid = os.getpid()

        log.debug('cgroup: %s will try to set initial cgroups %r', pid, cgroup_inherited)
        # do not write to log under privileged user
        log_str = '\n'
        with (UserPrivileges if has_root() else dummy)():
            for section, cgroup_name in cgroup_inherited.items():
                try:
                    tasks_path = os.path.join(cgroups_root, section, cgroup_name, 'tasks')
                    open(tasks_path, 'wb').write('%s\n' % (pid, ))
                    log_str += '\tcgroup: {} set inherited cgroup {}/{}\n'.format(pid, section, cgroup_name)
                except BaseException as ex:
                    log_str += '\tcgroup: {} failed to set inherited cgroup {}/{}: {}\n'.format(
                        pid, section, cgroup_name, ex
                    )
        log.debug(log_str)
        try:
            if cgroup is not None and os.path.isdir(cgroups_root):
                log.debug('cgroup: %s will try to set cgroup %r in all controllers', pid, cgroup)
                found_in_controllers = False

                log_str = '\n'
                with (dummy if uid is None else UserPrivileges)(peer_uid):
                    try:
                        for section in os.listdir(cgroups_root):
                            tasks_path = os.path.join(cgroups_root, section, cgroup, 'tasks')

                            if os.path.exists(tasks_path) and os.path.isfile(tasks_path):
                                found_in_controllers = True
                                try:
                                    open(tasks_path, 'wb').write('{}\n'.format(pid))
                                    log_str += '\tcgroup: {} set: cgroup {} section {}\n'.format(
                                        pid, cgroup, section
                                    )
                                except Exception as err:
                                    log_str += '\tcgroup: {} failed to set cgroup {} section {}: {}\n'.format(
                                        pid, cgroup, section, err
                                    )
                    except Exception as err:
                        log_str += '\tcgroup: {} failed to set cgroup {}: {}\n'.format(pid, cgroup, err)

                log.debug(log_str)
                if not found_in_controllers:
                    log.debug('cgroup: unable to find cgroup %r in any controller', cgroup)
        except BaseException as err:
            log.debug('cgroup: failed to set cgroup {}: {}'.format(cgroup, err))

        if uid is not None:
            with (UserPrivileges if has_root() else dummy)(limit=False):
                setResourceLimits(user_limits)
            UserPrivileges(uid, store=False, limit=False).__enter__()
        setlimits(limits)

    def func():
        os.execve(args[0], args, env)

    return procopen(prefunc, func, pty)


def procopen(prefunc, func, pty):
    fds = (os.pipe(), os.pipe(), os.pipe())
    try:
        hndl = fork(fds, prefunc, func, pty)
    except:
        for pipe in fds:
            os.close(pipe[0])
            os.close(pipe[1])
        raise

    (stdin, stdout, stderr) = fds

    os.close(stdin[0])
    os.close(stdout[1])
    os.close(stderr[1])

    return hndl, Fd(stdin[1]), Fd(stdout[0]), Fd(stderr[0])


def setlimits(limits):
    limits = build_limits(limits)
    for key, val in limits.items():
        new_key = getattr(resource, "RLIMIT_" + key.upper())
        resource.setrlimit(new_key, val)


class WaitPid(object):
    """
    Don't you ever dare to remove this and use other ways
    to handle SIGCHLD. pg@ claims that he managed to miss some signals
    using sighandler + waitpid(-1, WNOHANG).
    """
    def __init__(self):
        self.pids = {}
        daemonthr(self.run)

    def run(self):
        tout = 0

        while 1:
            try:
                mailbox().put(partial(self.onexit, *os.wait()))
                tout = 0
            except OSError:
                time.sleep(tout)
                tout = min((tout + 0.1) * 1.2, 1)
            except Exception as e:
                logging.getLogger('waitpid').normal(str(formatException(e)))

    def onexit(self, pid, status):
        e = self.event(pid)
        e.set(status)

    def wait(self, pid, timeout=None):
        e = self.event(pid)
        e.wait(timeout)
        del self.pids[pid]
        if e.ready():
            return e.get_nowait()
        return None

    def event(self, pid):
        if pid not in self.pids:
            self.pids[pid] = event.AsyncResult()
        return self.pids[pid]


@singleton
def pidwaiter():
    return WaitPid()


def fork(fds, prefunc, func, use_pty):
    rpipe, wpipe = os.pipe()

    def _safe_prefunc(prefunc=prefunc):
        try:
            prefunc()
        except Exception:
            import traceback
            msg = msgpack.dumps((False, traceback.format_exc()))
            os.write(wpipe, struct.pack('!I', len(msg)))
            os.write(wpipe, msg)
        else:
            msg = msgpack.dumps((True, ))
            os.write(wpipe, struct.pack('!I', len(msg)))
            os.write(wpipe, msg)

    (pid, master) = pty.fork() if use_pty else (os.fork(), -1)

    if not pid:
        # Close read pipe in child process
        os.close(rpipe)
        try:
            child(fds, _safe_prefunc, func, use_pty)
        finally:
            os._exit(-1)
    else:
        # Close write pipe in main process
        os.close(wpipe)

    try:
        msglen_raw = os.read(rpipe, 4)

        if msglen_raw and len(msglen_raw) == 4:
            msglen = struct.unpack('!I', msglen_raw)[0]

            buf = ''
            while len(buf) < msglen:
                data = os.read(rpipe, msglen - len(buf))
                if not data:
                    break
                buf += data

            if len(buf) == msglen:
                msg = msgpack.loads(buf)
                if not msg[0]:
                    raise Exception(msg[1])
            else:
                raise Exception('Unable to communicate with forked child (died before we received message)')
        else:
            raise Exception('Unable to communicate with forked child')

    finally:
        os.close(rpipe)

    return pid, Fd(master)


def child(fds, prefunc, func, pty):
    (stdin, stdout, stderr) = fds

    prefunc()

    # these are pty descriptors
    nfds = 2 + int(pty)
    fds = [os.dup(i) for i in six.moves.xrange(nfds)]

    # dup our pipe descriptors to stdin/stdout/stderr
    os.dup2(stdin[0], sys.stdin.fileno())
    os.dup2(stdout[1], sys.stdout.fileno())
    os.dup2(stderr[1], sys.stderr.fileno())

    # remap
    for i in six.moves.xrange(nfds):
        os.dup2(fds[i], i + nfds)
    os.closerange(2 * nfds, subprocess.MAXFD)

    # restore signal handlers
    signal.signal(signal.SIGHUP, signal.SIG_DFL)

    func()


def status_to_string(status):
    if os.WIFEXITED(status):
        return 'exited with code: %d' % (os.WEXITSTATUS(status), )

    if os.WIFSIGNALED(status):
        return 'exited by signal: %d' % (os.WTERMSIG(status), )

    if os.WCOREDUMP(status):
        return 'coredump'

    return ''


def status_to_int(ret, sig, coredumped):
    exitstatus = (ret << 8) | sig
    if coredumped:
        exitstatus |= 0x80
    return exitstatus


class FdHolder(object):
    def __init__(self, *fds):
        self.fds = fds

    def __enter__(self):
        return self.fds

    def __exit__(self, *args):
        for fd in self.fds:
            fd.close()


class BaseProcess(object):
    def runner(self, *args, **kwargs):
        raise NotImplementedError

    def executing(self):
        raise NotImplementedError

    def send_signal(self, signal):
        raise NotImplementedError

    def update_context(self, ctx):
        raise NotImplementedError

    def destroy_context(self):
        pass

    @classmethod
    def reconnect(cls, *args, **kwargs):
        raise NotImplementedError

    @property
    def repr(self):
        return None


class PortoProcess(BaseProcess):
    PROPS_WHITELIST = (
        'cpu_limit',
        'enable_porto',
        'memory_limit',
        'net_limit',
        'root',
        'root_readonly',
        'umask',
        'weak',
    )

    def __init__(self, parent, job=None):
        self.parent = parent
        self.log = parent.log
        self.job = job

        self.stream = parent.stream
        self.streamseq = 0
        self.streamlen = 0
        self.stdout_offset = -1
        self.stderr_offset = -1
        self.stdout = ""
        self.stderr = ""

        if job is not None:
            return

        container_name = None

        try:
            parent.log.debug('Creating porto container')
            conn = parent.parent.portoconn
            container_name = self._container_name(parent.root_container, parent.uuid)
            try:
                conn.Destroy(container_name)  # if it already exists
            except ContainerDoesNotExist:
                pass
            self.job = conn.Create(container_name)

            props = {k: v for k, v in parent.porto_options.iteritems()} if parent.porto_options else {}

            for k, v in props.iteritems():
                if parent.peer_uid != 0 and k not in self.PROPS_WHITELIST:
                    raise Exception('user requested to set banned porto property: %s' % (k,))
                self.job.SetProperty(k, v)

            userinfo = pwd.getpwuid(parent.uid if parent.uid is not None else geteuid())
            username = userinfo.pw_name
            group = grp.getgrgid(userinfo.pw_gid).gr_name

            for k, v in dict(isolate=False,  # TODO set by option  # noqa
                             respawn=bool(parent.keeprunning),
                             command=portoutils.build_command(parent.args),
                             stdin_path='/dev/null',
                             stdout_limit=parent.maxstreamlen,
                             user=username,
                             group=group,
                             owner_user=username,
                             owner_group=group,
                             cwd=parent.cwd,
                             ulimit=portoutils.build_limits_string(build_limits(parent.limits)),
                             env=portoutils.build_env(parent.env),
                             private='PROCMAN',
                             aging_time=600,
                             capabilities_ambient='',
                             ).iteritems():
                try:
                    self.job.SetProperty(k, v)
                except (InvalidProperty, UnknownError, NotSupported):
                    # old version returns UnknownError instead of InvalidProperty
                    pass

        except Exception as e:
            self.log.error("container creation failed: %s", traceback.format_exc())
            try:
                if self.job is not None:
                    self.job.Destroy()  # if it already exists
                elif container_name is not None:
                    conn.Destroy(container_name)
            except ContainerDoesNotExist:
                pass
            finally:
                self.job = None
            raise e

    @property
    def repr(self):
        if self.job:
            return "porto %s" % self.job.name

    def send_signal(self, num):
        self.job.Kill(num)

    def update_context(self, ctx):
        self.log.debug("Update context storage in the porto.")

        try:
            # we need to encode dumped str, because protobuf doesn't allow non-ascii
            # chars to present in non-unicode strings
            ctx = msgpack.dumps(ctx)
            ctx_filename = self.parent.parent.rundir.join(self.parent.uuid + '.porto').strpath
            with open_file(ctx_filename, 'wb') as f:
                f.write(ctx)
        except Exception as e:
            self.log.exception("cannot update porto context: %s (%s)", type(e), e, exc_info=sys.exc_info())
            raise errors.UpdateContextException(2, str(e))

    def destroy_context(self):
        try:
            ctx_filename = self.parent.parent.rundir.join(self.parent.uuid + '.porto').strpath
            os.unlink(ctx_filename)
        except Exception as e:
            self.log.warn("cannot remove context file: %s", e)
        try:
            self.job.Destroy()
        except Exception:
            self.log.warn("cannot destroy porto container: %s", e)

    def executing(self):
        return self.job and self.job.GetData("state") == "running"

    @staticmethod
    def _container_name(root_container, uuid):
        name = "procman-{}".format(uuid)
        if root_container is not None:
            name = '{}/{}'.format(root_container, name)
        return name

    def runner(self):
        try:
            state = self.job.GetData('state')
            if state == 'stopped':
                self.job.Start()
            elif state == 'dead' and self.parent.keeprunning:
                self.job.Stop()
                self.job.Start()
            elif state == 'paused':
                self.job.Resume()
        except EError as e:
            raise Exception("Cannot start porto container: %s" % (e,))

        while True:
            state = self.job.GetData('state')  # get early, check lately, so you lost no output
            self._check_output('stdout', 'OUT')
            self._check_output('stderr', 'ERR')
            if state == 'dead':
                break
            sleep(0.5)

        return int(self.job.GetData('exit_status'))

    def _check_output(self, outtype, label):
        offset = int(self.job.GetData(outtype + '_offset'))
        begin = getattr(self, outtype + '_offset')
        end = begin + len(getattr(self, outtype))
        if offset < begin:
            return

        if offset >= end:
            data = self.job.GetData("%s[%d]" % (outtype, offset))
            setattr(self, outtype + '_offset', offset)
        else:
            # read only diff
            data = self.job.GetData("%s[%d]" % (outtype, end))
            setattr(self, outtype + '_offset', end)

        setattr(self, outtype, data)
        for i in six.moves.xrange(0, len(data), 1024):
            self.stream.append((self.streamseq, (label, data[i:i + 1024])))
            self.streamseq += 1
        self.streamlen += len(data)

        while self.streamlen > self.parent.maxstreamlen:
            data = self.stream.popleft()
            self.streamlen -= len(data[1][1])

    @classmethod
    def reconnect(cls, Proc, server, container_info_path, nobody_uid):
        l = logging.getLogger('process')

        bn = os.path.splitext(os.path.basename(container_info_path))[0]
        if not server.portoconn:
            l.warn("[%s] Do not reconnecting: no porto connection", bn)
            return None

        try:
            with open_file(container_info_path, 'rb') as f:
                ctx = msgpack.load(f)
            if not isinstance(ctx, dict):
                raise ValueError("context isn't a dictionary")
        except (EnvironmentError,) + msgpack_exceptions as e:
            l.warn("[%s] Reconnect failed, unlinking info file: cannot read context file: %s", bn, e)
            os.unlink(container_info_path)
            return None

        container_name = cls._container_name(ctx['root_container'], ctx['uuid'])
        try:
            job = server.portoconn.Find(container_name)
        except ContainerDoesNotExist:
            l.warn('[%s] Connect failed: container already removed, unlinking info file', container_name)
            os.unlink(container_info_path)
            return None

        ctx['proc'] = False
        ctx['peer_pid'] = 0
        try:
            proc = Proc(server, nobody_uid=nobody_uid, **ctx)
            proc.proc = cls(proc, job)
        except Exception as e:
            if 'uuid' in ctx and ctx['uuid'] in server.procs:
                del server.procs[ctx['uuid']]
            l.warn('[%s] Cannot attach to existing container %s, unlinking info file: %s',
                   ctx.get('uuid'), job.name, e)
            os.unlink(container_info_path)
            return None
        proc.log.debug("Successfully connected")
        spawn(proc.run, partial(proc.proc.runner))
        return proc


class PidProcess(BaseProcess):
    def __init__(self, parent, pid):
        self.parent = parent
        self.log = parent.log
        self.pid = pid

        self.stream = parent.stream
        self.streamlen = 0
        self.streamseq = 0
        self.started = False

    def send_signal(self, num):
        if self.pid:
            with auto_user_privileges(self.parent.uid if self.parent.uid is not None else self.parent.nobody_uid):
                os.kill(self.pid, num)

    def executing(self):
        return not self.started or bool(self.pid)

    @property
    def repr(self):
        if self.pid:
            return "pid %d" % self.pid
        return None


class Subprocess(PidProcess):
    def __init__(self, parent):
        super(Subprocess, self).__init__(parent, None)
        self.loggedErrorStreamIndex = 0

    def runner(self):
        try:
            with auto_user_privileges(self.parent.uid if self.parent.uid is not None else self.parent.nobody_uid):
                (hndl, stdin, stdout, stderr) = runproc(
                    self.log,
                    self.parent.args,
                    self.parent.env, self.parent.cwd, self.parent.uid, self.parent.peer_uid, self.parent.cpus,
                    limits=self.parent.limits,
                    cgroup_inherited=self.parent.cgroup_inherited,
                    cgroup=self.parent.cgroup,
                )
            (self.pid, master) = hndl
        except BaseException:
            raise
        finally:
            self.started = True

        with FdHolder(stdin, stdout, stderr, master) as (stdin, stdout, stderr, master):
            self.log.info('spawned')

            if self.parent.logStderr:
                more_data = event.Event()

            def read(fd, label):
                while 1:
                    try:
                        part = gread(fd, 4096)
                    except:
                        break
                    if not part:
                        break

                    data = (self.streamseq, (label, part))
                    self.streamseq += 1
                    self.stream.append(data)
                    self.streamlen += len(part)

                    while self.stream and self.streamlen > self.parent.maxstreamlen:
                        popped = self.stream.popleft()
                        self.streamlen -= len(popped[1][1])

                    if self.parent.logStderr and label == 'ERR':
                        more_data.set()

            readers = (spawn(read, stdout, 'OUT'), spawn(read, stderr, 'ERR'))
            if self.parent.logStderr:
                def _stderr_logger():
                    active = True
                    need_log_something = False
                    while active:
                        try:
                            more_data.wait(timeout=(0.5 if need_log_something else None))
                            if more_data.isSet():
                                more_data.clear()
                                need_log_something = True
                                continue
                            more_data.clear()
                        except GreenletExit:
                            active = False
                            need_log_something = True

                        if not need_log_something:
                            continue

                        nomore = False
                        while not nomore:
                            data = []
                            datalen = 0
                            while True:
                                try:
                                    seq, (label, part) = self.stream[self.loggedErrorStreamIndex]
                                except IndexError:
                                    nomore = True
                                    break
                                self.loggedErrorStreamIndex += 1
                                if label != 'ERR':
                                    continue
                                data.append(part)
                                datalen += len(part)
                                if datalen > (10 << 10):
                                    break

                            if data:
                                self.log.warning('stderr: %s', ''.join(data))

                        need_log_something = False

                stderr_logger = spawn(_stderr_logger)
                readers[1].link(lambda grn: stderr_logger.kill())

            status = pidwaiter().wait(self.pid)
            stoptime = time.time()

            assert status is not None, 'status with infinite timeout should be not None!'

            if self.parent.logStderr:
                stderr_logger.join(timeout=10)

            self.log.info('exited (status %d: %s)', status, status_to_string(status))

            if os.WIFSIGNALED(status) and os.WCOREDUMP(status) and self.parent.cdumpcfg:
                try:
                    self.parent.core = self.parent.cdumpcfg.collect(
                        self.parent.get_context(), self.log,
                        self.parent.cwd,
                        self.parent.uid if self.parent.uid is not None else self.parent.nobody_uid,
                        self.pid,
                        self.parent.uuid,
                        stoptime,
                    )
                except Exception:
                    self.parent.core = None
                    self.log.warning("core collection failed")

            self.pid = None
            return status

    def update_context(self, ctx):
        pass  # plain subprocess has no context and we cannot reconnect to it anyway

    @classmethod
    def reconnect(cls, *args, **kwargs):
        pass


class LinerProcess(PidProcess):
    class LinerVersionMismatch(Exception):
        pass

    LINER_VERSION = 1           # Current liner version.
    LINER_START_TIMEOUT = 120   # Wait till the control socket will be created by the liner.
    LINER_CONNECT_TIMEOUT = 10  # Wait for liner response.

    def __init__(self, parent, pid, lpid, liner):
        super(LinerProcess, self).__init__(parent, pid)
        self.lpid = lpid
        self.liner = liner
        self._pending_signal = None

        # for liner_runner this is the pid we need to wait for if liner exits
        # we dont need to wait any process for adopted liner, but if we dont wait
        # started liner -- we will have defunct process
        self.waitpid = None

    def executing(self):
        return super(LinerProcess, self).executing() or bool(self.lpid)

    def runner(self, sock=None):
        sockreader = None
        sockname = self.parent.parent.rundir.join(self.parent.uuid).strpath
        if not self.pid:
            try:
                self.lpid = None
                self.log.debug('Starting liner (%r) on the socket %r.', liner(), sockname)

                with auto_user_privileges(self.parent.uid if self.parent.uid is not None else self.parent.nobody_uid):
                    ((self.lpid, master), stdin, stdout, stderr) = runproc(
                        self.log,
                        [liner(), sockname],
                        self.parent.env, self.parent.cwd, self.parent.uid, self.parent.peer_uid, self.parent.cpus,
                        pty=False,
                        limits=self.parent.limits,
                        cgroup_inherited=self.parent.cgroup_inherited,
                        cgroup=self.parent.cgroup,
                    )
                self.waitpid = self.lpid
            finally:
                self.started = True

            try:
                with FdHolder(stdin, stdout, stderr, master):  # as (stdin, stdout, stderr, master):
                    self.log.debug('Liner (PID %d) spawned.', self.lpid)
                    sock, sockreader = self._connect_liner_with_timeout(sockname)
            except BaseException:
                # we failed to connect to liner
                # reset lpid to notify user that process is not running anymore
                self.lpid = None
                raise

            # Now we can close liner's out/err pipes and switch into normal functional state.
            self.liner = sock
            self.parent._update_context()
        else:
            self.log.debug("Existing liner adopting.")
            # Connecting already running liner
            sockreader = self._socket_reader(sock)
            # Switch connected liner into the normal mode
            sock.send(msgpack.dumps("adopt"))
            self.liner = sock
            self.started = True
            # I wanna to be a paranoid and update liner's context here,
            # BUT, it can be a zombie actually, which already sent a message with exit status
            # and terminated normally. So we have to go socket messages read here first.

        return self._process_liner_messages(self.liner, sockreader)

    def update_context(self, ctx):
        if not self.liner or not isinstance(self.liner, socket.socket):
            return

        self.log.debug("Update context storage in the liner.")

        try:
            self._send({'context': msgpack.dumps(ctx)})
        except Exception as e:
            raise errors.UpdateContextException(2, str(e))

    def send_signal(self, num):
        if self.pid:
            super(LinerProcess, self).send_signal(num)
        else:
            self._pending_signal = num

    def _send(self, msg):
        self.liner.send(msgpack.dumps(msg))

    @staticmethod
    def _socket_reader(sock, timeout=None):
        sock.settimeout(timeout)
        unpacker = msgpack.Unpacker()
        while True:
            buf = sock.recv(0x100)
            if not buf:
                raise Exception('Liner unexpectedly closed the connection.')
            unpacker.feed(buf)
            for o in unpacker:
                yield o

    @staticmethod
    def _connect_liner(name, log):
        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        have_root = has_root()
        # Connect socked as root in case we can do it .)
        timeout = 60
        log.debug('Connect socket %r as %s (timout %ds)', name, 'root' if have_root else 'current user', timeout)
        old_timeout = sock.gettimeout()
        sock.settimeout(timeout)
        with auto_user_privileges():
            sock.connect(name)
        sock.settimeout(old_timeout)
        return sock

    def _connect_liner_with_timeout(self, sockname):
        wait_for = self.LINER_START_TIMEOUT
        while wait_for >= 0:
            if os.path.exists(sockname):
                try:
                    sock = self._connect_liner(sockname, self.log)
                except Exception as ex:
                    self.log.debug('Failed to connect liner: %s', ex)
                else:
                    break
            if os.waitpid(self.lpid, os.WNOHANG)[0]:
                # liner process exited, so we have to throw exception
                raise Exception('Liner process exited before we connected to socket.')
            wait_for -= .05
            sleep(.05)
        else:
            # loop was not breaked -- we failed to connect
            raise Exception('Liner start timed out.')

        sockreader = self._socket_reader(sock)
        self.log.debug('Liner connected. Starting sub-process.')
        sock.send(msgpack.dumps('info'))
        assert sockreader.next()['version'] == self.LINER_VERSION

        sock.send(msgpack.dumps(['%s (%s)' % (' '.join(self.parent.tags), self.parent.uuid)] + self.parent.args))
        self.pid = sockreader.next()
        self.log.info('Sub-process PID is %d. Going to normal functional mode.', self.pid)
        if self._pending_signal:
            self.log.info('Send pending signal %d to %d.', self._pending_signal, self.pid)
            super(LinerProcess, self).send_signal(self._pending_signal)
            self._pending_signal = None

        return sock, sockreader

    def _process_liner_messages(self, sock, sockreader):
        try:
            for data in sockreader:
                for label in ('OUT', 'ERR'):
                    key = 'stdout' if label == 'OUT' else 'stderr'
                    chunk = data.get(key, None)
                    if not chunk:
                        continue
                    self.streamseq += 1
                    self.stream.append((self.streamseq, (label, chunk)))
                    self.streamlen += len(chunk)

                    while self.stream and self.streamlen > self.parent.maxstreamlen:
                        popped = self.stream.popleft()
                        self.streamlen -= len(popped[1][1])

                    if label == 'ERR' and self.parent.logStderr:
                        self.log.warning(' stderr: %s', chunk.strip())

                if 'exited' in data:
                    stoptime = time.time()
                    # Process process termination event
                    self.log.info('exited %r', data)
                    if data.get('coredump') and self.parent.cdumpcfg:
                        self.parent.core = self.parent.cdumpcfg.collect(
                            self.parent.get_context(),
                            self.log,
                            self.parent.cwd,
                            self.parent.uid if self.parent.uid is not None else self.parent.nobody_uid,
                            self.pid,
                            self.parent.uuid,
                            stoptime,
                        )

                    # Tell liner to terminate itself and his process group
                    self.log.debug('Sending terminatepg command to liner')
                    sock.send(msgpack.dumps('terminatepg'))
                    return status_to_int(data['exitstatus'], data.get('termsig', 0), data.get('coredump'))
        except BaseException as ex:
            excinfo = sys.exc_info()
            try:
                self.log.error("Unhandled exception caught: %s: %s", ex.__class__.__name__, ex)
            except:
                pass

            # Since we possible have communication error -- try to forcibly send
            # terminatepg to liner. If we dont do this -- liner communication loop may stuck forever, while
            # liner will be function normally. In that case we will wait for liner exit forever in finally block.
            tpg_tries = 60
            for i in range(tpg_tries):
                try:
                    self.log.debug('Sending terminatepg command to liner')
                    sock.send(msgpack.dumps('terminatepg'))
                except BaseException as ex:
                    if isinstance(ex, socket.error) and ex.errno == errno.EPIPE:
                        # Probably liner died. We will not be able to connect him after procman reboot anyway
                        # so, just post an error and continue
                        break

                    try:
                        self.log.error('Failed with: %s: %s', ex.__class__.__name__, ex)
                    except:
                        pass

                    if i == (tpg_tries - 1):
                        # We are not able to kill liner gracefully. There is nothing to do here, just
                        # emergency exit :(
                        self.log.critical('Unable to stop liner normally, emergency exit')
                        os._exit(1)
                    else:
                        sleep(1)
                else:
                    break

            six.reraise(*excinfo)
        finally:
            if self.waitpid:
                self.log.debug('Waiting liner process for exit (pid: %d)', self.waitpid)
                status = pidwaiter().wait(self.waitpid)
                self.log.debug('Liner process exited with status %r', status)
            else:
                self.log.debug('Dont waiting process exit for adopted liner')

            self.log.debug('Waiting liner socket to be closed')

            def _waitsockclose(timeout):
                with Timeout(timeout) as timeout_instance:
                    try:
                        assert sock.recv(1) == ''
                        return True
                    except Timeout as timeout_instance_received:
                        if timeout_instance_received != timeout_instance:
                            raise
                return False

            if not _waitsockclose(60):
                self.log.error('Liner not exited normally, killing his process group')
                for i in range(60):
                    if _waitsockclose(1):
                        self.log.debug('Liner socket closed')
                        break
                else:
                    self.log.critical('Failed to kill liner!!!')
            else:
                self.log.debug('Liner socket closed')

            self.log.debug("Liner runner loop finished.")
            self.liner = True
            self.lpid = None
            self.pid = None

    @classmethod
    def reconnect(cls, Proc, server, sockname, nobody_uid):
        l = logging.getLogger('process')
        try:
            sock = cls._connect_liner(sockname, l)
        except Exception as ex:
            l.warn("Connect failed: %s. Unlinking socket file...", str(ex))
            os.unlink(sockname)
            return None

        try:
            try:
                peer_user = pwd.getpwuid(getpeerid(sock)[0]).pw_name
            except (EnvironmentError, AttributeError):
                return None

            sock.send(msgpack.dumps('info'))
            reader = cls._socket_reader(sock, cls.LINER_CONNECT_TIMEOUT)

            info = reader.next()
            if info['version'] != cls.LINER_VERSION:
                raise cls.LinerVersionMismatch('We want %r we got %r' % (
                    cls.LINER_VERSION, info['version']
                ))
            ctx = msgpack.loads(info['context'])
            ctx['proc'] = False  # Do not spawn a sub-process
            ctx['nobody_uid'] = nobody_uid

            # security fix: ignore 'user' returned by liner, use real running user id
            ctx['user'] = peer_user

            try:
                proc = Proc(server, **ctx)
                proc.proc = cls(proc, info['pid'], info['pgid'], sock)
            except:
                if 'uuid' in ctx and ctx['uuid'] in server.procs:
                    del server.procs[ctx['uuid']]
                raise
            proc.log.debug("Successfully connected")
            spawn(proc.run, partial(proc.proc.runner, sock=sock))
            return proc
        except Exception as ex:
            if isinstance(ex, cls.LinerVersionMismatch):
                l.warning('Liner version mismatch: %s', ex)
            else:
                l.error('Got unhandled exception while reconnecting liner: %s', traceback.format_exc())

            l.debug('Sending terminatepg and waiting for 5 seconds for socket to be closed')

            sock.send(msgpack.dumps('terminatepg'))
            with Timeout(5) as timeout:
                try:
                    while sock.recv(1) != '':

                        pass
                    l.info('Old liner exited!')
                except Timeout as ex:
                    if ex != timeout:
                        raise
                    l.error('Old liner refuses to die =(. Will leave him orphan, removing sock file')
                    os.unlink(sockname)

            sock.close()
            return None
