from cStringIO import StringIO
from porto.exceptions import SocketError, SocketTimeout
import os
import sys
import resource
import functools
import socket
import types
import gevent.socket
import gevent

from .utils import auto_user_privileges

__all__ = ['build_command', 'build_env', 'build_limits_string', 'PortoAdapter']

_NetworkExceptions = (socket.error, SocketError, SocketTimeout)


def _quote_str(s):
    io = StringIO()
    io.write("'")
    for char in s:
        if char == "'":
            io.write("'\"'\"'")
        else:
            io.write(char)
    io.write("'")
    return io.getvalue()


def build_env(env=None):
    if env is None:
        env = {k: v for k, v in os.environ.iteritems() if k in ('SHELL', 'TERM', 'PATH', 'PYTHONPATH')}

    return '; '.join(
        '{}={}'.format(k, v.replace(';', '\;'))
        for k, v in env.iteritems()
    )


def build_command(args):
    first = True
    io = StringIO()

    for arg in args:
        if not first:
            io.write(' ')
        io.write(_quote_str(arg))
        first = False

    return io.getvalue()


def _limit_value(lim):
    return "unlim" if lim == resource.RLIM_INFINITY else str(lim)


def build_limits_string(limits):
    return '; '.join(
        "{}: {} {}".format(lim, _limit_value(soft), _limit_value(hard))
        for lim, (soft, hard)
        in limits.iteritems()
    )


class PortoAdapter(object):
    def __init__(self, slave, log=None):
        self.__slave = slave
        self.__log = log

    @staticmethod
    def _is_porto(obj):
        if 'porto.api' not in getattr(obj, '__module__', ''):
            return False
        if not hasattr(obj, '__class__'):
            return False
        if '_RPC' == obj.__class__.__name__:
            return False
        return True

    def __wrap(self, item):
        if callable(item):
            return self.__wrapper(item)
        elif self._is_porto(item):
            return PortoAdapter(item)
        elif isinstance(item, (list, types.GeneratorType)):
            return [
                (PortoAdapter(x) if self._is_porto(x) else x)
                for x in item
            ]
        else:
            return item

    def __try_connect(self):
        slave = self.__slave
        if hasattr(slave, 'conn'):
            slave = slave.conn
        if not hasattr(slave, 'rpc'):
            return
        if slave.rpc.sock is None:
            slave.TryConnect()

    def __wrapper(self, cb):
        @functools.wraps(cb)
        def _fn(*args, **kwargs):
            for attempt in range(3):
                try:
                    with auto_user_privileges():
                        self.__try_connect()
                    result = cb(*args, **kwargs)
                    return self.__wrap(result)
                except _NetworkExceptions:
                    if self.__log is not None:
                        self.__log.warning(
                            "porto call %s failed: ",
                            cb, exc_info=sys.exc_info()
                        )
                    if attempt == 2:
                        raise
                    gevent.sleep(1.0)
        return _fn

    def __getattr__(self, item):
        res = getattr(self.__slave, item)
        return self.__wrap(res)
