import collections
import copy
import errno
import logging
import os
import pwd
import signal
import py
import time
import traceback
import sys

from gevent import spawn, sleep

from .kernel_util.functional import singleton
from .kernel_util.logging import MessageAdapter
from .kernel_util.uuid import genuuid
from .kernel_util.errors import saveTraceback

from . import execution
from .coredump import CoreCollectorConfig
from .utils import has_root


@singleton
def getenviron():
    good = frozenset(['TERM', 'LANG', 'LD_LIBRARY_PATH', 'LANG', 'LC_ALL', 'SHELL', 'PYTHONPATH', 'PATH'])

    return dict(filter(lambda x: x[0] in good, os.environ.items()))


def parsesignum(num):
    return getattr(signal, num)


class Proc(object):
    LINER_VERSION = 1           # Current liner version.
    LINER_START_TIMEOUT = 10    # Wait 10 seconds till the control socket will be created by the liner.
    # Context to be stored together with liner process. They will be passed back to the constructor on cotext restore.
    CONTEXT_SLOTS = (
        'peer_pid peer_uid '
        'args env user tags keeprunning cwd maxstreamlen afterlife logStderr '
        'cpus uuid cdumpcfg limits cgroup cgroup_inherited '
        'porto root_container porto_options'.split()
    )

    def __init__(  # noqa
        self,
        parent,
        peer_pid,    # pid of asking to run process
        peer_uid,    # uid of asking to run process
        nobody_uid,  # nobody user uid
        args=[],
        env={},
        user=None,
        tags=[],
        keeprunning=True,
        cwd='/',
        maxstreamlen=64 * 1024,
        afterlife=120,
        logStderr=False,
        cpus=None,
        uuid=None,
        liner=None,
        cdumpcfg=None,
        fdlimit=None,
        limits=None,
        cgroup=None,
        cgroup_inherited=None,
        porto=None,
        root_container=None,
        porto_options=None,
        proc=None,
    ):
        fullenv = getenviron().copy()

        # Dirty hack for cygwin: liner sometimes hangs for unknown reason
        # and we have to live with it somehow, so we do not use liner at all
        if sys.platform == 'cygwin':
            liner = None

        for (key, value) in env.items():
            fullenv[str(key)] = str(value)

        self.proc = None
        self.parent = parent
        self.cwd = cwd
        self.uuid = genuuid() if not uuid else uuid
        self.args = [str(arg) for arg in args]
        self.env = fullenv
        self.tags = tags if isinstance(tags, list) else list(tags)
        self.maxrespawntimeout = 10

        self.keeprunning = keeprunning
        self.afterlife = afterlife
        self.logStderr = logStderr
        self.cpus = cpus
        self.core = None

        self.uid = None
        self.nobody_uid = nobody_uid
        self.user = user
        self.liner = liner
        self.cgroup = cgroup
        self.cgroup_inherited = cgroup_inherited
        self.cdumpcfg = None
        self.porto = porto
        self.portojob = None
        self.root_container = root_container
        self.porto_options = porto_options
        if cdumpcfg:
            self.cdumpcfg = cdumpcfg.copy() if not isinstance(cdumpcfg, dict) else CoreCollectorConfig(**cdumpcfg)
            if self.cdumpcfg.folder is not None:
                self.cdumpcfg.folder = py.path.local(self.cdumpcfg.folder)
                self.cdumpcfg.user_folder = True
            else:
                self.cdumpcfg.folder = self.parent.rundir.join('coredumps')
                self.cdumpcfg.user_folder = False

        if limits is None:
            limits = {}
        climit = self.cdumpcfg.size if self.cdumpcfg and self.cdumpcfg.size else None
        if climit is not None:
            limits["core"] = climit
        if fdlimit is not None:
            limits["nofile"] = fdlimit
        self.limits = limits

        if user:
            if not has_root():
                raise EnvironmentError(errno.EPERM, 'Don`t have root privileges')

            try:
                pw_rec = pwd.getpwnam(user)
            except KeyError:
                raise EnvironmentError(errno.EPERM, 'No user `{0}` on system'.format(user))
            else:
                self.uid = pw_rec.pw_uid

            if (peer_uid != 0) and (peer_uid != self.uid):
                raise EnvironmentError(
                    errno.EPERM, 'Can`t start process from user `{0}` (peer uid is `{1}`)'.format(self.uid, peer_uid)
                )
        else:
            raise EnvironmentError(
                errno.EPERM, 'Procman is unable to run unprivileged processes anymore, set "user" param'
            )

        self.peer_uid = peer_uid
        self.peer_pid = peer_pid

        # statuses
        self.results = []
        self.execErrors = []

        # process stdout/err
        self.stream = collections.deque()
        self.maxstreamlen = maxstreamlen

        # register proc
        parent.procs[self.uuid] = self

        try:
            self.log = MessageAdapter(
                logging.getLogger('process'),
                fmt='[%(uuid)s : %(tags)s] %(message)s',
                data={
                    'uuid': str(self.uuid),
                    'tags': '.'.join(sorted(tags))
                }
            )

            if porto and not self.parent.portoconn:
                raise EnvironmentError(errno.EINVAL, "Porto is not available")

            self.log.info("Preparing to start %s", self)
            # liner variable can be:
            #   function, False : if liner greenlet already started
            #   None            : if we dont need liner here
            #   True            : if we need to start liner here
            if proc is False:
                return  # process is already started, no spawn is needed

            if proc:
                self.proc = proc
            elif porto:
                self.proc = execution.PortoProcess(self)
                spawn(self.run, self.proc.runner)
            elif liner:
                self.proc = execution.LinerProcess(self, None, None, None)
                spawn(self.run, self.proc.runner)
            else:
                self.proc = execution.Subprocess(self)
                spawn(self.run, self.proc.runner)
            self._update_context()
        except Exception:
            parent.procs.pop(self.uuid, None)
            raise

    @classmethod
    def reconnect(cls, parent, nobody_uid, sockname=None, container_info_path=None):
        if sockname is not None:
            return execution.LinerProcess.reconnect(cls, parent, sockname, nobody_uid)
        elif container_info_path is not None:
            return execution.PortoProcess.reconnect(cls, parent, container_info_path, nobody_uid)

    def _update_context(self):
        ctx = self.get_context()
        ctx['cdumpcfg'] = self.cdumpcfg.to_dict() if self.cdumpcfg else None
        if self.proc and self.proc.repr:
            self.proc.update_context(ctx)

    def run(self, runner):
        try:
            while True:
                if not self.proc or not self.proc.repr:
                    argsstr = ' '.join([a if ' ' not in a else "'%s'" % (a, ) for a in self.args])
                    self.log.info('start process %s' % (argsstr, ))
                    self.log.debug('cwd: %s, env: %r, cpus: %r', self.cwd, self.env, self.cpus)

                starttime = time.time()
                self.results.append(runner())

                if self.keeprunning:
                    cur = time.time()
                    duration = cur - starttime
                    rtout = self.maxrespawntimeout / pow(1 + duration, 0.5)
                    self.log.info('wait %f seconds before respawn', rtout)
                    deadline = cur + rtout

                    while self.keeprunning and time.time() < deadline:
                        sleep(0.1)

                if not self.keeprunning:
                    break
        except BaseException as e:
            self.log.error("Unhandled exception: %s", traceback.format_exc())
            saveTraceback(e)
            self.execErrors.append(e)
            raise
        finally:
            try:
                self.stoptime = time.time()
                if self.afterlife:
                    self.log.info('finished')
                    sleep(self.afterlife)
                else:
                    self.log.info('finished and complete')
            finally:
                del self.parent.procs[self.uuid]
                if self.proc:
                    self.proc.destroy_context()
                if self.afterlife:
                    self.log.info('complete')

    def stop_retries(self):
        self.keep_running(False)

    def executing(self):
        return self.proc and self.proc.executing()

    def running(self):
        return not hasattr(self, 'stoptime')

    def keep_running(self, val):
        self.log.info('"keep running" changed (%d -> %d)', int(self.keeprunning), int(val))
        self.keeprunning = val
        self._update_context()

    def send_signal(self, num):
        return self.signal(num)

    def signal(self, num):
        if not self.executing():
            self.log.warning('skip sending signal - process not running')
            return
        if isinstance(num, basestring):
            num = parsesignum(num)
            if num is None:
                raise Exception("invalid signal %s" % num)
        self.log.info('sending signal %d', num)
        try:
            self.proc.send_signal(num)
        except EnvironmentError as err:
            self.log.warning('failed to deliver signal: %s', err.strerror)
        except Exception as err:
            self.log.warning('failed to deliver signal: %s', str(err))

    def kill(self):
        self.stop()

    def stop(self):
        # doesn't work from api - Procs() dispatches it to self
        self.signal('SIGKILL')

    def terminate(self):
        self.signal('SIGTERM')

    def __repr__(self):
        if self.proc and self.proc.repr:
            return '(%s, uuid %s, tags %s)' % (self.proc.repr, self.uuid, self.tags)
        return '%s' % self.uuid

    def add_tags(self, tags):
        self.tags.extend(tags)
        self.log.debug('Adding process tag(s) %r', tags)
        self._update_context()

    def delete_tags(self, tags):
        self.tags[:] = [_ for _ in self.tags if _ not in tags]
        self.log.debug('Deleting process tag(s) %r', tags)
        self._update_context()

    def get_tags(self):
        return frozenset(self.tags)

    def get_context(self):
        return dict((k, copy.deepcopy(getattr(self, k))) for k in self.CONTEXT_SLOTS)

    # Old API support
    getContext = get_context
    getTags = get_tags
    deleteTags = delete_tags
    addTags = add_tags
    keepRunning = keep_running
    stopRetries = stop_retries

    def stat(self):
        ret = {
            'retcodes': [execution.status_to_string(status) for status in self.results],
            'errors': list(self.execErrors),
            'stream': list(self.stream),
        }

        for x in ('cwd', 'uuid', 'args', 'env', 'tags', 'start', 'stoptime', 'core'):
            ret[x] = getattr(self, x, None)

        # FIXME
        for x in ('pid', 'lpid'):
            ret[x] = getattr(getattr(self, 'proc', None), x, None)
        ret['porto_container'] = getattr(getattr(getattr(self, 'proc', None), 'job', None), 'name', None)

        return ret
