import sys
import six
from functools import partial

from kernel.pyro.client import PyroProxy
from api.procman import (
    ProcManException, ProcessLookupException, UpdateContextException,
    INVALID_NAME, RUNNING, FINISHED
)


__all__ = ['ProcManException',
           'ProcessLookupException',
           'UpdateContextException',
           'INVALID_NAME', 'RUNNING', 'FINISHED',
           'ProcessProxy', 'ProcMan']


def _wrap_exception():
    e_class, e_val, e_tb = sys.exc_info()

    if e_class.__name__ in (
        'ProcessLookupException',
        'UpdateContextException',
    ):
        e_new_class = globals()[e_class.__name__]
        e_val.__class__ = e_new_class
        six.reraise(e_new_class, e_val, e_tb)

    raise


class RemoteCallWrapper(object):
    def __init__(self, call):
        self.call = call

    def __getattr__(self, val):
        return getattr(self.call, val)

    def __call__(self, *args, **kwargs):
        try:
            return self.call(*args, **kwargs)
        except Exception:
            _wrap_exception()


class ProcessProxy(object):
    def __init__(self, pm, handle):
        self.pm = pm
        self.handle = str(handle)

    def __str__(self):
        return self.handle

    def __getattr__(self, name):
        return RemoteCallWrapper(partial(getattr(self.pm, name), self.handle))

    def stream(self, flt=''):
        ret = six.moves.cStringIO()
        for (seq, (label, data)) in self.stat()['stream']:
            if label.startswith(flt):
                ret.write(data)

        return ret.getvalue()

    def stdout(self):
        return self.stream('OUT')

    def stderr(self):
        return self.stream('ERR')

    def status(self):
        try:
            if self.executing():
                return RUNNING
            return FINISHED
        except Exception:
            return INVALID_NAME


class ProcMan(object):
    def __init__(self, uri):
        self.__server_uri = uri
        self.__server = None

    def server(self):
        if self.__server is None:
            retries = [1, 2, 3, 4, 10, 20, 20]
            self.__server = PyroProxy(self.__server_uri, retries=retries)
            self.__server.maySendRequest = lambda: True
        return self.__server

    def create(self, *args, **kwargs):
        return ProcessProxy(self, self.server().create(*args, **kwargs))

    def from_string(self, str):
        if self.is_valid(str):
            return ProcessProxy(self, str)
        return None

    def adapted(self, l):
        return [ProcessProxy(self, name) for name in l]

    def find_by_tags(self, tags, excludeTags=None):  # noqa
        return self.adapted(self.server().find_by_tags(tags, excludeTags))

    def find_by_tags_entry(self, tags):
        return self.adapted(self.server().find_by_tags_entry(tags))

    def enumerate(self):
        return [ProcessProxy(self, name) for name in self.server().enumerate()]

    def find_by_pid(self, pid):
        try:
            return ProcessProxy(self, self.server().find_by_pid(pid))
        except Exception:
            _wrap_exception()

    def find_by_uuid(self, uuid):
        try:
            return ProcessProxy(self, self.server().find_by_uuid(uuid))
        except Exception:
            _wrap_exception()

    # Old API support
    findByTags = find_by_tags
    findByPid = find_by_pid
    findByUUID = find_by_uuid

    def __getattr__(self, name):
        return getattr(self.server(), name)


def procman_constructor(uri):
    return partial(ProcMan, uri)
