import contextlib
import resource
import logging
import pickle
import os
import stat
import sys

from contextlib import contextmanager

import gevent.socket
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros

from .kernel_util.functional import singleton, memoized
from .kernel_util.logging import renameLevels as rename_logging_levels  # noqa
from .kernel_util.sys.user import UserPrivileges as user_privileges  # noqa


def build_limits(limits):
    valid = {}
    for key, lim in limits.iteritems():
        if isinstance(lim, tuple):
            soft, hard = lim
        else:
            soft = hard = lim

        if soft == sys.maxint or soft is None:
            soft = resource.RLIM_INFINITY
        if hard == sys.maxint or hard is None:
            hard = resource.RLIM_INFINITY

        try:
            new_key = "RLIMIT_" + key.upper()
            getattr(resource, new_key)
        except AttributeError:
            raise ValueError("Invalid limit name: {}".format(key))
        else:
            valid[key] = (soft, hard)

    return valid


def name():
    return 'procman'


@singleton
def liner():
    return os.path.join(os.path.dirname(sys.executable), 'liner')


@contextmanager
def dummy(*args, **kwargs):
    yield


def gread(fd, buf):
    fd = fileno(fd)
    gevent.socket.wait_read(fd)
    return os.read(fd, buf)


class Fd(object):
    def __init__(self, fd):
        self.__fd = fd

    def fileno(self):
        assert self.valid()

        return self.__fd

    def close(self):
        if self.valid():
            try:
                close = os.close
            except AttributeError:
                return

            close(self.fileno())
            self.__fd = -1

    def valid(self):
        return self.__fd >= 0

    def __del__(self):
        self.close()


def fileno(fd):
    if isinstance(fd, (int, long)):
        return fd
    return fd.fileno()


@memoized
def has_root():
    if os.uname()[0].lower() == 'darwin' or os.uname()[0].lower().startswith('cygwin'):
        return getuid() == 0
    else:
        return getresuid()[2] == 0


_privileges_lock = coros.RLock()


@contextlib.contextmanager
def auto_user_privileges(*args, **kwargs):
    if has_root():
        with _privileges_lock, user_privileges(modifyGreenlet=False, *args, **kwargs):
            yield
    else:
        yield


def geteuid():
    with _privileges_lock:
        return os.geteuid()


def getuid():
    with _privileges_lock:
        return os.getuid()


def getresuid():
    with _privileges_lock:
        return os.getresuid()


def getgid():
    with _privileges_lock:
        return os.getgid()


def open_file(*args, **kwargs):
    with _privileges_lock:
        return open(*args, **kwargs)


# TODO: fixme in cygwin python
def _chown(path, uid, gid):
    if sys.platform == 'cygwin':
        old_path = path.strpath
        path.strpath = path.strpath.replace('/', '\\')
        path.chown(uid, gid)
        path.strpath = old_path
    else:
        path.chown(uid, gid)


def fixperms(rundir, logdir, cdumpdir):
    uid = geteuid()  # skynet

    with auto_user_privileges():
        for path in rundir, logdir, cdumpdir:
            if not path.check(dir=1):
                if path.check(exists=1) or path.check(link=1):
                    path.remove()
                path.ensure(dir=1)

            # change owner if needed
            if has_root() and (path.stat().uid != uid or path.stat().gid != 0):
                _chown(path, uid, 0)

        # 0o1777 on rundir and cdumpdir (or, if it fails -- 0o777)
        for stickydir in (rundir, cdumpdir):
            try:
                stickydir.chmod(stat.S_ISVTX | stat.S_IRWXU | stat.S_IRWXG | stat.S_IRWXO)
            except EnvironmentError:
                # FreeBSD can deny sticky bit for nonroot
                # We do allow 777 here since it works only in case of non-root + dir doesn't exist
                stickydir.chmod(stat.S_IRWXU | stat.S_IRWXG | stat.S_IRWXO)

        # 0o755 on logdir
        logdir.chmod(stat.S_IRWXU | stat.S_IRGRP | stat.S_IXGRP | stat.S_IROTH | stat.S_IXOTH)

        # 0o644 on logfile
        for logfile in logdir.listdir():
            if logfile.basename.startswith('procman.log'):
                _chown(logfile, uid, logfile.stat().gid)
                logfile.chmod(stat.S_IRUSR | stat.S_IWUSR | stat.S_IRGRP | stat.S_IROTH)


def setup_logger(logdir, max_bytes, backup_count):
    rename_logging_levels(3)

    root_log = logging.getLogger('')
    root_log.setLevel(logging.DEBUG)

    handler = logging.handlers.RotatingFileHandler(
        logdir.join('procman.log').strpath,
        maxBytes=max_bytes,
        backupCount=backup_count
    )

    pid = os.getpid()

    handler.setFormatter(logging.Formatter('%5d  %%(asctime)s:  %%(levelname)s  %%(message)s' % (pid, )))
    handler.setLevel(logging.DEBUG)
    root_log.addHandler(handler)

    if sys.stdout.isatty():
        handler = logging.StreamHandler(sys.stdout)
        handler.setFormatter(logging.Formatter('%(asctime)s  %(levelname)s  %(message)s'))
        handler.setLevel(logging.DEBUG)
        root_log.addHandler(handler)

    return root_log.getChild('main')


class SafeUnpickler(pickle.Unpickler):
    def find_class(self, module, name):
        raise ImportError("non-POD objects are forbidden")

    def load_persid(self):
        raise TypeError("non-POD objects are forbidden")

    load_inst = load_obj = load_newobj = load_global = load_binpersid = load_persid
    load_ext1 = load_ext2 = load_ext4 = load_reduce = load_persid
