#include "router_api.h"

#include <infra/libs/service_iface/str_iface.h>
#include <util/generic/yexception.h>
#include <util/string/cast.h>

namespace NProtoLogger {
    TString BuildClientPath(const TString& clientName, EHandlerTypeFlag handlerType) {
        return TString::Join("/", clientName, ToString(handlerType));
    }

    template<typename TReq, typename TRsp>
    void AddClientRoutes(TSimpleSharedPtr<NInfra::TRequestRouter<NProtoLogger::IApi>> router, const TString& clientName, EHandlerType handlerTypes) {
        if (handlerTypes.HasFlags(EHandlerTypeFlag::PROTO)) {
            (*router).template Add<NInfra::TSerializedProtoRequest<TReq>, NInfra::TSerializedProtoReply<TRsp>>(BuildClientPath(clientName, EHandlerTypeFlag::PROTO), TString::Join("Write to ", clientName), &IApi::Write2Log);
        }
        if (handlerTypes.HasFlags(EHandlerTypeFlag::JSON)) {
            (*router).template Add<NInfra::TJson2ProtoRequest<TReq>, NInfra::TProto2JsonReply<TRsp>>(BuildClientPath(clientName, EHandlerTypeFlag::JSON), TString::Join("Json write to", clientName), &IApi::Write2Log);
        }
        if (handlerTypes.HasFlags(EHandlerTypeFlag::JSON_BATCH)) {
            (*router).template Add<NInfra::TJsonBatch2ProtoRequest<TReq>, NInfra::TProtoBatch2JsonReply<TRsp>>(BuildClientPath(clientName, EHandlerTypeFlag::JSON_BATCH), TString::Join("Json write batch to", clientName), &IApi::WriteBatch2Log);
        }
    }

    NInfra::TRequestRouterPtr CreateRouter(IApi& service, const TProtoLoggerConfig&) {
        auto router = MakeSimpleShared<NInfra::TRequestRouter<IApi>>(service);

        (*router)
            .Add<NInfra::TEmptyRequest<NApi::TReqPing>, NInfra::TProto2JsonReply<NApi::TRspPing>>("/ping", "Ping", &IApi::Ping)
            .Add<NInfra::TEmptyRequest<NApi::TReqShutdown>, NInfra::TEmptyReply<NApi::TRspShutdown>>("/shutdown", "Shutdown", &IApi::Shutdown)
            .Add<NInfra::TEmptyRequest<NApi::TReqReopenLog>, NInfra::TProto2JsonReply<NApi::TRspReopenLog>>("/reopen_log", "Reopen log", &IApi::ReopenLog);

        AddClientRoutes<NApi::TReqWrite2LogQuasar, NApi::TRspWrite2LogQuasar>(router, "quasar", EHandlerTypeFlag::ALL);
        AddClientRoutes<NApi::TReqWriteMetrics2LogQuasar, NApi::TRspWrite2LogQuasar>(router, "quasar_metrics", EHandlerTypeFlag::ALL);
        AddClientRoutes<NApi::TReqCardShownLog, NApi::TRspCardShownLog>(router, "searchapp_morda_metrics", EHandlerTypeFlag::JSON);

        return router;
    }
}
