#include "service.h"

#include <infra/proto_logger/libs/logger/events/events.ev.pb.h>
#include <infra/proto_logger/libs/router_api/router_api.h>

#include <library/cpp/protobuf/util/pb_io.h>
#include <library/cpp/watchdog/watchdog.h>

#include <util/datetime/base.h>
#include <util/generic/serialized_enum.h>
#include <util/string/cast.h>
#include <util/stream/file.h>
#include <util/string/subst.h>
#include <util/system/hostname.h>
#include <util/system/mlock.h>

namespace NProtoLogger {
    TService::TService(const TProtoLoggerConfig& config)
        : Config_(config)
        , HttpService_(Config_.GetHttpServiceConfig(), CreateRouter(*this, Config_))
        , EventLog_(Config_.GetEventLogConfig())
        , Logger_(Config_.GetLoggerConfig())
        , StatisticsClient_(Config_.GetStatisticsClientConfig(), EventLog_)
    {
    }

    void TService::Start() {
        OpenLogs(Config_);
        BuildClientDropRatios(Config_);
        HttpService_.Start(EventLog_.SpawnFrame());
    }

    void TService::Wait() {
        HttpService_.Wait(EventLog_.SpawnFrame());
    }

    void TService::Ping(NInfra::TRequestPtr<NApi::TReqPing>, NInfra::TReplyPtr<NApi::TRspPing> reply) {
        NApi::TRspPing result;
        result.SetData(DEFAULT_RSP_PING);
        reply->Set(result);
    }

    void TService::Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown>, NInfra::TReplyPtr<NApi::TRspShutdown>) {
        this->Shutdown();
    }

    void TService::ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog>, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) {
        NApi::TRspReopenLog result;
        result.SetData(DEFAULT_RSP_REOPEN_LOG);
        reply->Set(result);
        EventLog_.ReopenLog();
        Logger().ReopenLogs();
        StatisticsClient_.ReopenLogs();
    }

    void TService::Write2Log(NInfra::TRequestPtr<NApi::TReqWrite2LogQuasar> request, NInfra::TReplyPtr<NApi::TRspWrite2LogQuasar> reply) {
        Write2LogImpl(request, reply);
    }

    void TService::WriteBatch2Log(NInfra::TRequestPtr<TVector<NApi::TReqWrite2LogQuasar>> requests, NInfra::TReplyPtr<TVector<NApi::TRspWrite2LogQuasar>> reply) {
        WriteBatch2LogImpl(requests, reply);
    }

    void TService::Write2Log(NInfra::TRequestPtr<NApi::TReqWriteMetrics2LogQuasar> request, NInfra::TReplyPtr<NApi::TRspWrite2LogQuasar> reply) {
        Write2LogImpl(request, reply);
    }

    void TService::WriteBatch2Log(NInfra::TRequestPtr<TVector<NApi::TReqWriteMetrics2LogQuasar>> requests, NInfra::TReplyPtr<TVector<NApi::TRspWrite2LogQuasar>> reply) {
        WriteBatch2LogImpl(requests, reply);
    }

    void TService::Write2Log(NInfra::TRequestPtr<NApi::TReqCardShownLog> request, NInfra::TReplyPtr<NApi::TRspCardShownLog> reply) {
        NUdpClickMetrics::NApi::TReqIncreaseClickMetrics metricsRequest;
        *metricsRequest.MutableSearchappMordaMetrics() = request->Get();
        StatisticsClient_.IncreaseMetrics(metricsRequest);
        NApi::TRspCardShownLog result;
        result.SetResponseStatus(DEFAULT_RSP_CARDSHOWNLOG);
        reply->Set(result);
    }

    template <typename TReq>
    void TService::Write2LogImpl(NInfra::TRequestPtr<TReq> request, NInfra::TReplyPtr<NApi::TRspWrite2LogQuasar> reply) {
        TReq message = request->Get();
        const TString& clientLogPath = GetClientLogPath(request->Path(), message);
        const float dropRatio = GetClientDropRatio(request->Path());
        message.SetServerTimestamp(Now().Seconds());
        const auto& result = WriteMessage2LogImpl(message, clientLogPath, dropRatio);
        reply->Set(result);
    }

    template <typename TReq>
    void TService::WriteBatch2LogImpl(NInfra::TRequestPtr<TVector<TReq>> requests, NInfra::TReplyPtr<TVector<NApi::TRspWrite2LogQuasar>> reply) {
        NApi::TRspWrite2LogQuasar badResult;
        badResult.SetResponseStatus(SERIALIZATION_ERROR_MESSAGE);

        const TString& path = requests->Path();
        bool allBatchRequestsBroken = !requests->Get().empty();

        ui64 timestamp = Now().Seconds();
        TVector<NApi::TRspWrite2LogQuasar> result;
        for (TReq request : requests->Get()) {
            try {
                const TString& clientLogPath = GetClientLogPath(path, request);
                const float dropRatio = GetClientDropRatio(path);

                request.SetServerTimestamp(timestamp);
                const auto& writeResult = WriteMessage2LogImpl(request, clientLogPath, dropRatio);
                result.emplace_back(writeResult);
                allBatchRequestsBroken = false;
            } catch (const TSerializationError&) {
                result.emplace_back(badResult);
            }
        }

        if (allBatchRequestsBroken) {
            ythrow TBatchSerializationError() << BATCH_SERIALIZATION_ERROR_MESSAGE;
        }

        reply->Set(result);
    }

    void TService::OpenLogs(const TProtoLoggerConfig& config) {
        Logger().OpenLogs(BuildClientLogPaths(config));
    }

    void TService::Shutdown() {
        static THolder<IWatchDog> abortWatchDog = THolder<IWatchDog>(CreateAbortByTimeoutWatchDog(CreateAbortByTimeoutWatchDogOptions(TDuration::Minutes(1)), "Ooops!"));
        HttpService_.ShutDown();
    }

    template <>
    NApi::TRspWrite2LogQuasar TService::WriteMessage2LogImpl(
        const NApi::TReqWriteMetrics2LogQuasar& request,
        const TString& clientLogPath,
        float dropRatio
    ) {
        try {
            NUdpClickMetrics::NApi::TReqIncreaseClickMetrics metricsRequest;
            *metricsRequest.MutableQuasarMetrics() = request;
            StatisticsClient_.IncreaseMetrics(metricsRequest);
        } catch (...) {
            EventLog_.SpawnFrame()->LogEvent(ELogPriority::TLOG_ERR, NEv::TMetricsSendError(CurrentExceptionMessage()));
        }
        return WriteMessage2LogImpl<::google::protobuf::Message>(request, clientLogPath, dropRatio);
    }

    template <typename TMessage>
    NApi::TRspWrite2LogQuasar TService::WriteMessage2LogImpl(
        const TMessage& request,
        const TString& clientLogPath,
        float dropRatio
    ) {
        NApi::TRspWrite2LogQuasar result;
        if (dropRatio && RandomNumber<float>() < dropRatio) {
            result.SetResponseStatus(RSP_DROPPED);
        } else if (TString output; NProtoBuf::TrySerializeToBase64String(request, output)) {
            Logger().Log(clientLogPath).AddLog("%s\n", output.data());
            result.SetResponseStatus(DEFAULT_RSP_WRITE2LOG);
        } else {
            ythrow TSerializationError() << SERIALIZATION_ERROR_MESSAGE;
        }
        return result;
    }

    const TString& TService::GetClientLogPathImpl(const TString& path) const {
        const TString& clientLogPath = ClientPathToLogPath_.at(path);
        if (Logger().IsLoggingQueueFull(clientLogPath)) {
            ythrow TLoggerQueueLimitReachedError() << LOGGER_QUEUE_LIMIT_REACHED_ERROR_MESSAGE;
        }
        return clientLogPath;
    }

    template <typename TReq>
    const TString& TService::GetClientLogPath(const TString& path, const TReq& request) const {
        Y_UNUSED(request);
        return GetClientLogPathImpl(path);
    }

    template <>
    const TString& TService::GetClientLogPath(const TString& path, const NApi::TReqWriteMetrics2LogQuasar& request) const {
        const auto& metricName = request.GetMetricName();
        if (metricName == "minidump") {
            return GetClientLogPathImpl(TString::Join(path, "-", metricName));
        }
        return GetClientLogPathImpl(path);
    }

    void TService::BuildClientLogPath(
        const TString& clientName,
        const TString& logPath,
        const TString& port,
        const TString& extraLog,
        TVector<TString>* clientLogPaths
    ) {
        const TString& dashedExtraLog = extraLog ? TString::Join("-", extraLog) : extraLog;
        const TString& clientLogPath = TString::Join(logPath, "current-eventlog-", clientName, dashedExtraLog, "-", port);
        for (const auto& handlerType : GetEnumAllValues<EHandlerTypeFlag>()) {
            ClientPathToLogPath_.insert({TString::Join(BuildClientPath(clientName, handlerType), dashedExtraLog), clientLogPath});
        }
        clientLogPaths->emplace_back(clientLogPath);
    }

    TVector<TString> TService::BuildClientLogPaths(const TProtoLoggerConfig& config) {
        TVector<TString> clientLogPaths;
        const TString& port = ToString(config.GetHttpServiceConfig().GetPort());
        const TString& logPath = config.GetLogPath();
        for (const auto& client: config.GetClients()) {
            const TString& clientName = client.GetName();
            BuildClientLogPath(clientName, logPath, port, "", &clientLogPaths);
            for (const auto& extraLog: client.GetExtraLogs()) {
                BuildClientLogPath(clientName, logPath, port, extraLog, &clientLogPaths);
            }
        }
        return clientLogPaths;
    }

    void TService::BuildClientDropRatios(const TProtoLoggerConfig& config) {
        for (const auto& client : config.GetClients()) {
            Y_ENSURE(0 <= client.GetDropRatio() && client.GetDropRatio() <= 1, "Drop ratio option must be in [0, 1] interval");
            for (const auto& handlerType : GetEnumAllValues<EHandlerTypeFlag>()) {
                ClientPathToDropRatio_.insert({BuildClientPath(client.GetName(), handlerType), client.GetDropRatio()});
            }
        }
    }

    float TService::GetClientDropRatio(const TString& path) const {
        return ClientPathToDropRatio_.at(path);
    }
} // namespace NProtoLogger
