#pragma once

#include <infra/libs/http_service/service.h>
#include <infra/libs/service_iface/errors.h>
#include <infra/libs/udp_metrics/client/client.h>
#include <infra/proto_logger/libs/config/config.pb.h>
#include <infra/proto_logger/libs/logger/logger.h>
#include <infra/proto_logger/libs/service_iface/service_iface.h>
#include <infra/udp_click_metrics/api/api.pb.h>

namespace NProtoLogger {
    namespace {
        const TString SERIALIZATION_ERROR_MESSAGE = "Protobuf API couldn't serialize data in request";
        const TString BATCH_SERIALIZATION_ERROR_MESSAGE = "Protobuf API couldn't serialize any request in batch";
        const TString LOGGER_QUEUE_LIMIT_REACHED_ERROR_MESSAGE = "Logger queue size limit reached, can not log data in request";
        const TString DEFAULT_RSP_PING = "Pong";
        const TString DEFAULT_RSP_REOPEN_LOG = "OK";
        const TString DEFAULT_RSP_WRITE2LOG = "OK";
        const TString DEFAULT_RSP_CARDSHOWNLOG = "OK";
        const TString RSP_DROPPED = "DROPPED";
    }

    class TSerializationError : public NInfra::TServiceError {
    public:
        TSerializationError()
            : TServiceError(HttpCodes::HTTP_BAD_REQUEST, grpc::StatusCode::INVALID_ARGUMENT)
        {
        }
    };

    class TBatchSerializationError : public NInfra::TServiceError {
    public:
        TBatchSerializationError()
            : TServiceError(HttpCodes::HTTP_BAD_REQUEST, grpc::StatusCode::INVALID_ARGUMENT)
        {
        }
    };

    class TLoggerQueueLimitReachedError : public NInfra::TServiceError {
    public:
        TLoggerQueueLimitReachedError()
            : TServiceError(HttpCodes::HTTP_SERVICE_UNAVAILABLE, grpc::StatusCode::UNAVAILABLE)
        {
        }
    };

    class TService: public IService {
    public:
        TService(const TProtoLoggerConfig& config);
        ~TService() = default;

        void Start() override;

        void Wait() override;

        void Ping(NInfra::TRequestPtr<NApi::TReqPing> request, NInfra::TReplyPtr<NApi::TRspPing> reply) override;

        void Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown> request, NInfra::TReplyPtr<NApi::TRspShutdown> reply) override;

        void ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog> request, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) override;

        void Write2Log(NInfra::TRequestPtr<NApi::TReqWrite2LogQuasar> request, NInfra::TReplyPtr<NApi::TRspWrite2LogQuasar> reply) override;

        void WriteBatch2Log(NInfra::TRequestPtr<TVector<NApi::TReqWrite2LogQuasar>> requests, NInfra::TReplyPtr<TVector<NApi::TRspWrite2LogQuasar>> reply) override;

        void Write2Log(NInfra::TRequestPtr<NApi::TReqWriteMetrics2LogQuasar> request, NInfra::TReplyPtr<NApi::TRspWrite2LogQuasar> reply) override;

        void WriteBatch2Log(NInfra::TRequestPtr<TVector<NApi::TReqWriteMetrics2LogQuasar>> requests, NInfra::TReplyPtr<TVector<NApi::TRspWrite2LogQuasar>> reply) override;

        void Write2Log(NInfra::TRequestPtr<NApi::TReqCardShownLog> request, NInfra::TReplyPtr<NApi::TRspCardShownLog> reply) override;

        TLogger& Logger() noexcept {
            return Logger_;
        }

        const TLogger& Logger() const noexcept {
            return Logger_;
        }

        void OpenLogs(const TProtoLoggerConfig& config);

        void Shutdown();

    private:
        template<typename TReq>
        void Write2LogImpl(NInfra::TRequestPtr<TReq> request, NInfra::TReplyPtr<NApi::TRspWrite2LogQuasar> reply);

        template<typename TReq>
        void WriteBatch2LogImpl(NInfra::TRequestPtr<TVector<TReq>> requests, NInfra::TReplyPtr<TVector<NApi::TRspWrite2LogQuasar>> reply);

        template<typename TMessage>
        NApi::TRspWrite2LogQuasar WriteMessage2LogImpl(const TMessage& request, const TString& clientLogPath, float dropRatio);

        template<typename TReq>
        const TString& GetClientLogPath(const TString& path, const TReq& request) const;

        const TString& GetClientLogPathImpl(const TString& path) const;

        TVector<TString> BuildClientLogPaths(const TProtoLoggerConfig& config);

        void BuildClientLogPath(const TString& clientName, const TString& logPath, const TString& port, const TString& extraLog , TVector<TString>* clientLogPaths);

        void BuildClientDropRatios(const TProtoLoggerConfig& config);

        float GetClientDropRatio(const TString& path) const;

    private:
        const TProtoLoggerConfig& Config_;
        NInfra::THttpService HttpService_;
        NInfra::TLogger EventLog_;
        TLogger Logger_;
        NUdpMetrics::TSelfBalancingClient<NUdpClickMetrics::NApi::TReqIncreaseClickMetrics> StatisticsClient_;

        TMap<TString, TString> ClientPathToLogPath_;
        THashMap<TString, float> ClientPathToDropRatio_;
    };
} // namespace NProtoLogger
