import datetime
import os
import time
import logging
import multiprocessing
import psutil
import pytz

import ctrl
import utils


def calc_cpu_limit(guarantee, host_total_usage, psi_usage, high_watermark):
    left = high_watermark - host_total_usage
    return max(50.0, guarantee, min(high_watermark, left + psi_usage))


def _is_night():
    t = datetime.datetime.now(pytz.timezone('Europe/Moscow')).time()
    return datetime.time(hour=0, minute=0) <= t <= datetime.time(hour=8, minute=30)


def _is_disabled():
    return os.path.exists('flags/disable_dynamic_cpu')


class CpuUsageMonitor(object):
    def __init__(self, probe_interval, window_size):
        self._usage = utils.MovingAverage(probe_interval, window_size)

    def update(self):
        raise RuntimeError()

    @property
    def usage(self):
        return self._usage.value

    @property
    def enough_history(self):
        return self._usage.enough_history


class ContainerCpuUsage(CpuUsageMonitor):
    def __init__(self, name, probe_interval, window_size):
        super(ContainerCpuUsage, self).__init__(probe_interval, window_size)
        self._name = name
        self._prev = None
        self._last_upd = None

    def update(self):
        with utils.porto_connect() as connection:
            now = time.time()
            ns = long(connection.GetProperty(self._name, 'cpu_usage'))
            if self._prev and ns > self._prev:
                self._usage.push(100.0 * (ns - self._prev) / (now - self._last_upd) / 1e9)
            self._prev = ns
            self._last_upd = now
        _log.debug('container usage: %s', self.usage)


class HostCpuUsage(CpuUsageMonitor):
    def update(self):
        value = psutil.cpu_percent() * multiprocessing.cpu_count()
        self._usage.push(value)
        _log.debug('host usage: %s', self.usage)


class CpuController(ctrl.Controller):
    def __init__(
        self,
        container_name,
        probe_interval,
        window_size,
        high_watermark,
        cpu_affinity,
        night_mode
    ):
        self._container_name = container_name
        self._probe_interval = probe_interval
        self._host_cpu_monitor = HostCpuUsage(probe_interval, window_size)
        self._container_cpu_monitor = ContainerCpuUsage(container_name, probe_interval, window_size)
        self._high_watermark = high_watermark
        self._cpu_guarantee = 0
        self._cpu_limit = 0
        self._slot_container_name = container_name.split('/', 1)[0]
        self._cpu_set = ''
        self._cpu_affinity = cpu_affinity
        self._night_mode = night_mode

        self.sleep_seconds = probe_interval

    def loop_iteration(self):
        self._host_cpu_monitor.update()
        self._container_cpu_monitor.update()
        with utils.porto_connect() as conn:
            self._cpu_guarantee = utils.cores_to_percent(conn.GetProperty(self._container_name, 'cpu_guarantee'))
            self._cpu_limit = utils.cores_to_percent(conn.GetProperty(self._container_name, 'cpu_limit'))

            if _is_disabled():
                conn.SetProperty(self._container_name, 'cpu_limit', utils.percent_to_cores(self._cpu_guarantee))
            elif self._host_cpu_monitor.enough_history and self._container_cpu_monitor.enough_history:
                high_watermark = self._high_watermark * multiprocessing.cpu_count()
                if self._night_mode and not _is_night():
                    high_watermark = 0

                new_limit = calc_cpu_limit(
                    self._cpu_guarantee,
                    self._host_cpu_monitor.usage,
                    self._container_cpu_monitor.usage,
                    high_watermark,
                )
                conn.SetProperty(self._container_name, 'cpu_limit', utils.percent_to_cores(new_limit))
                if new_limit != self._cpu_limit:
                    _log.info('set cpu_limit %s -> %s', self._cpu_limit, new_limit)

                if self._cpu_affinity:
                    cpu_set = utils.limit_to_affinity(new_limit / 100.0, psutil.cpu_count(logical=True))
                    conn.SetProperty(self._slot_container_name, 'cpu_set', cpu_set)
                    if cpu_set != self._cpu_set:
                        _log.info('set cpu_set on [%s] %s -> %s', self._slot_container_name, self._cpu_set, cpu_set)
                        self._cpu_set = cpu_set

    def signals(self):
        def _sig(signal_name):
            return '{}_ammv'.format(signal_name)
        return [
            [_sig('host_cpu_usage'), self._host_cpu_monitor.usage],
            [_sig('host_cpu_total'), multiprocessing.cpu_count() * 100],
            [_sig('yt_cpu_usage'), self._container_cpu_monitor.usage],
            [_sig('yt_cpu_guarantee'), self._cpu_guarantee],
            [_sig('yt_cpu_limit'), self._cpu_limit],
        ]


_log = logging.getLogger(__name__)
