import gevent.monkey
gevent.monkey.patch_all()  # noqa

import os
import time
import logging
import logging.handlers
import argparse

import cpu
import memory
import limits
import utils


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--debug', help='Debug mode (dump logs to stdout)', action='store_true', default=False)
    parser.add_argument('--container-regexp', help='container regexp', required=True)
    parser.add_argument('--cluster', help='cluster name (for signals)', required=True)
    parser.add_argument('--configs-url', help='configs url (for dynamic memory guarantee)', required=False)
    parser.add_argument('--high-watermark', help='desired cpu load [0..100]', required=True, type=float)
    parser.add_argument('--probe-interval', default=5, type=float)
    parser.add_argument('--window-size', default=60, type=float)
    parser.add_argument('--affinity', help='Setup cpu affinity', action='store_true', default=False)
    parser.add_argument('--port', type=int, required=False)
    parser.add_argument('--night-mode', help='Setup night mode (work time is 00:00-08:30 msk)', action='store_true', default=False)
    return parser.parse_args()


def setup_logging(debug):
    if not os.path.exists('./logs'):
        os.makedirs('./logs')

    formatter = SkynetishFormatter()

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    debug_handler = logging.handlers.RotatingFileHandler('./logs/debug.log', maxBytes=1024**3, backupCount=10)
    debug_handler.setLevel(logging.DEBUG)
    debug_handler.setFormatter(formatter)
    logger.addHandler(debug_handler)

    info_handler = logging.handlers.RotatingFileHandler('./logs/instance.log', maxBytes=1024**3, backupCount=10)
    info_handler.setLevel(logging.INFO)
    info_handler.setFormatter(formatter)
    logger.addHandler(info_handler)

    if debug:
        console_handler = logging.StreamHandler()
        console_handler.setLevel(logging.DEBUG)
        console_handler.setFormatter(formatter)
        logger.addHandler(console_handler)

    logging.getLogger('requests').setLevel(logging.WARNING)


class SkynetishFormatter(logging.Formatter):
    def __init__(self):
        super(SkynetishFormatter, self).__init__()

    def formatTime(self, record, **kwargs):
        t = time.strftime('%Y-%m-%d %H:%M:%S', self.converter(record.created))
        return '%s.%03d' % (t, record.msecs)

    def format(self, record):
        levelno = record.levelno
        if levelno > 5:
            level = '[%-4s]' % logging.getLevelName(levelno)
        else:
            level = '(%s)' % (str(levelno) if levelno < 0 else ' %d' % levelno)

        date = self.formatTime(record)
        message = record.getMessage()
        if record.exc_info:
            message += '\n' + self.formatException(record.exc_info)
        header = '{0} {1} [{2}]  '.format(date, level, record.name)

        if '\n' in message:
            # special case for multi-line log messages
            message_lines = message.splitlines()
            line = [header + message_lines[0]]
            prepend = '%s%s' % (' ' * (len(header) - 2), ': ')
            line.extend(['%s%s' % (prepend, l) for l in message_lines[1:]])
            line = '\n'.join(line)
        else:
            line = '{header}{message}'.format(header=header, message=message)
        return line


def main():
    args = parse_args()
    setup_logging(args.debug)
    container_name = utils.find_container(args.container_regexp)
    logging.info('resolved container: %s', container_name)

    sub_ctrls = [
        cpu.CpuController(
            container_name,
            args.probe_interval,
            args.window_size,
            args.high_watermark,
            args.affinity,
            args.night_mode
        ),
    ]
    if args.configs_url:
        sub_ctrls += [
            memory.MemoryController(container_name, args.cluster, args.configs_url)
        ]

    ctrl = limits.MainCtrl(args.cluster, sub_ctrls)
    if args.port:
        ctrl.spawn_yasm(args.port)
    ctrl.loop()


if __name__ == '__main__':
    main()
