import re
import math
import contextlib

import porto


@contextlib.contextmanager
def porto_connect():
    connection = None
    try:
        connection = porto.Connection()
        yield connection
    finally:
        if connection:
            connection.disconnect()


def cores_to_percent(cpu):
    cpu = float(cpu[:-1]) * 100
    return cpu


def percent_to_cores(number):
    cpu = number / 100.0
    return bytes(cpu) + b'c'


def limit_to_affinity(cores, ncpu):
    # ncpu=8; threads: 0 1 2 3
    #                  4 5 6 7
    # if cores = 3 or 4, take 2-3,5-6
    # if cores = 16 and ncpu = 56, take 20-27, 48-55
    # if cores = 56 and ncpu = 56, take 0-27, 28-55
    assert cores <= ncpu

    half_cores = int(math.ceil(cores / 2.0))
    half_ncpu = ncpu / 2
    return '{}-{},{}-{}'.format(half_ncpu - half_cores, half_ncpu - 1, ncpu - half_cores, ncpu - 1)


class MovingAverage(object):
    def __init__(self, probe_interval, window_size, value=0.0):
        self._coefficient = 1 - math.exp(-probe_interval / float(window_size))
        self._value = float(value)
        self._probes_count = 0
        self._probes_needed = window_size / probe_interval

    def push(self, value):
        self._value = self._coefficient * value + (1 - self._coefficient) * self._value
        self._probes_count += 1
        return self._value

    @property
    def value(self):
        return self._value

    @property
    def enough_history(self):
        return self._probes_count >= self._probes_needed


def find_container(name_regexp):
    pattern = re.compile(name_regexp)
    with porto_connect() as connection:
        lst = [
            name for name in connection.List()
            if pattern.match(name)
            and connection.GetProperty(name, 'state') in ('running', 'meta')
        ]
        if len(lst) == 1:
            return lst[0]
    raise RuntimeError('{} containers match regexp: {}'.format(len(lst), name_regexp))
