import os
import time
import json
import logging
import requests
import cityhash
from collections import namedtuple


def parse_api_conf(json_data):
    api = ApiConfig(**json_data['api'])
    return api


def parse_limits_conf(json_data):
    return LimitsConfig(
        mem=MemLimits(**json_data['limits']['mem']), cpu=CpuLimits(**json_data['limits']['cpu']),
        net=NetLimits(**json_data['limits']['net']),
    )


def parse_storages_conf(json_data):
    lst = []
    for storage in json_data['storages']:
        lst.append(StorageConfig(
            host=storage['host'],
            vm=storage['vm'],
            space=storage['space'],
            clusters=tuple(storage.get('clusters', []))
        ))
    return lst


def parse_root_volume_conf(json_data):
    if 'root_volume' in json_data:
        return RootVolumeConfig(
            json_data['root_volume']['space'],
            json_data['root_volume'].get('storage_path'),
            json_data['root_volume'].get('volume_path', os.getcwd()),
        )
    return None


def parse_porto_props(json_data):
    return json_data.get('porto', {})


def parse_conf(data):
    api = parse_api_conf(data)
    limits = parse_limits_conf(data)
    trigger = TriggerConfig(**data['trigger'])
    policy = StopPolicyConfig(**data['stop_policy'])
    root_volume = parse_root_volume_conf(data)
    storages = parse_storages_conf(data)

    config = _Config(
        namespace=parse_namespace(data),
        api=api,
        limits=limits,
        trigger=trigger,
        stop_policy=policy,
        root_volume=root_volume,
        storages=storages,
        vlans=data.get('vlans'),
        layers=data.get('layers'),
        command=data.get('command'),
        porto_props=parse_porto_props(data),
    )
    return config


def parse_namespace(json_data):
    return json_data['namespace']


def load_conf(path):
    with open(path) as f:
        data = json.load(f)
    return data


ApiConfig = namedtuple('ApiConfig', ['url', 'type', 'gencfg_ip', 'rate'])
TriggerConfig = namedtuple('TriggerConfig', ['interval', 'window_size', 'high_watermark', 'low_watermark'])
StorageConfig = namedtuple('StorageConfig', ['host', 'vm', 'space', 'clusters'])
StopPolicyConfig = namedtuple('StopPolicyConfig', ['drop_after', 'stop_after', 'destroy_on_stop_service'])
LimitsConfig = namedtuple('LimitsConfig', ['mem', 'cpu', 'net'])
CpuLimits = namedtuple('CpuLimits', ['null', 'destroy_on'])
MemLimits = namedtuple('MemLimits', ['null', 'destroy_on'])
NetLimits = namedtuple('NetLimits', ['total'])
RootVolumeConfig = namedtuple('RootVolumeConfig', ['space', 'storage_path', 'volume_path'])


class _Config(object):
    def __init__(
        self, namespace, api, limits, trigger,
        stop_policy, root_volume=None, storages=None, vlans=None,
        layers=None, command=None,
        porto_props=None,
    ):
        self.api = api
        self.namespace = namespace
        self.limits = limits
        self.trigger = trigger
        self.stop_policy = stop_policy

        self.vlans = vlans
        self.layers = _get_layers_md5(layers)
        self.root_volume = root_volume
        self.storages = storages
        self.command = command
        self.porto_props = porto_props

    def recalc_md5(self):
        self.layers = _get_layers_md5(self.layers)

    def __repr__(self):
        return """
            namespace: {}
            api: {}
            limits: {}
            trigger: {}
            stop_policy: {}
            root_volume: {}
            storages: {}
            vlans: {}
            layers: {}
            command: {}
            porto_props: {}
        """.format(
            self.namespace, self.api, self.limits, self.trigger, self.stop_policy,
            self.root_volume, self.storages, self.vlans, self.layers, self.command,
            self.porto_props,
        )


def load_power(url, hostname):
    while True:
        try:
            return requests.get('{}/resources/{}'.format(url, hostname)).json()['cpu']['power']
        except requests.RequestException as ex:
            logging.exception(ex)
            time.sleep(5)


def _get_layers_md5(layers):
    result = {}
    if layers:
        for cluster in layers:
            result[cluster] = []
            for layer in layers[cluster]:
                result[cluster].append({
                    'md5': cityhash.filehash64(layer['name']),
                    'name': layer['name']
                })
    return result
