import gevent.monkey
gevent.monkey.patch_all()

import argparse
import logging
import logging.handlers
import signal
import time
from utils import os_utils
from utils import names
from global_config import load_conf, parse_conf, load_power
from service import Service
import status_script
import notify_script
import install_script
import stop_script


def run_service(options):
    if options.status_script:
        status_script.main(options.port, options.config_path)
    elif options.notify_script:
        notify_script.main(options.port, options.config_path, options.changes)
    elif options.install_script:
        install_script.main()
    elif options.stop_script:
        stop_script.main(options.port)
    else:
        conf = parse_conf(load_conf(options.config_path))
        logging.info(conf)
        service = Service(options.port, conf, load_power(conf.api.url, names.hostname()))
        signal.signal(signal.SIGTERM, lambda signum, frame: service.stop())
        try:
            service.run()
        except KeyboardInterrupt:
            service.stop()


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--port', type=int)
    parser.add_argument('--debug', help='Debug mode (dump logs to stdout)', action='store_true', default=False)
    parser.add_argument('--config_path', help='path to config', required=True)
    parser.add_argument('--status_script', action='store_true', default=False)
    parser.add_argument('--notify_script', action='store_true', default=False)
    parser.add_argument('--install_script', action='store_true', default=False)
    parser.add_argument('--stop_script', action='store_true', default=False)
    parser.add_argument('--changes', default='')
    return parser.parse_args()


def setup_logging(debug):
    os_utils.ensure_dir('./logs')

    formatter = SkynetishFormatter()

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    debug_handler = logging.handlers.RotatingFileHandler('./logs/debug.log', maxBytes=1024**3, backupCount=10)
    debug_handler.setLevel(logging.DEBUG)
    debug_handler.setFormatter(formatter)
    logger.addHandler(debug_handler)

    info_handler = logging.handlers.RotatingFileHandler('./logs/instance.log', maxBytes=1024**3, backupCount=10)
    info_handler.setLevel(logging.INFO)
    info_handler.setFormatter(formatter)
    logger.addHandler(info_handler)

    if debug:
        console_handler = logging.StreamHandler()
        console_handler.setLevel(logging.DEBUG)
        console_handler.setFormatter(formatter)
        logger.addHandler(console_handler)

    logging.getLogger('requests').setLevel(logging.WARNING)


class SkynetishFormatter(logging.Formatter):
    def __init__(self):
        super(SkynetishFormatter, self).__init__()

    def formatTime(self, record, **kwargs):
        t = time.strftime('%Y-%m-%d %H:%M:%S', self.converter(record.created))
        return '%s.%03d' % (t, record.msecs)

    def format(self, record):
        levelno = record.levelno
        if levelno > 5:
            level = '[%-4s]' % logging.getLevelName(levelno)
        else:
            level = '(%s)' % (str(levelno) if levelno < 0 else ' %d' % levelno)

        date = self.formatTime(record)
        message = record.getMessage()
        if record.exc_info:
            message += '\n' + self.formatException(record.exc_info)
        header = '{0} {1} [{2}]  '.format(date, level, record.name)

        if '\n' in message:
            # special case for multi-line log messages
            message_lines = message.splitlines()
            line = [header + message_lines[0]]
            prepend = '%s%s' % (' ' * (len(header) - 2), ': ')
            line.extend(['%s%s' % (prepend, l) for l in message_lines[1:]])
            line = '\n'.join(line)
        else:
            line = '{header}{message}'.format(header=header, message=message)
        return line


def main():
    args = parse_args()
    setup_logging(args.debug)

    run_service(args)


if __name__ == '__main__':
    main()
