import sys
import gevent
import logging

from status.configs import OnlineConfig
from status.monitor import Monitor
from status.trigger import Trigger
from status.reporter import Reporter
from status.soloyasm import Server
from vm.controller import VmController


class Service(object):
    _sleep_time = 15
    _wait_till_success = 10
    _logger = logging.getLogger('Service')

    def __init__(self, port, conf, cpu_power):
        self._global_config = conf
        self._configs = OnlineConfig(conf)
        self._monitor = Monitor(conf.namespace, conf.vlans, conf.trigger.window_size)
        self._trigger = Trigger(
            conf.trigger.interval, conf.trigger.window_size,
            conf.trigger.low_watermark, conf.trigger.high_watermark
        )
        self._reporter = Reporter(
            conf.api.url, conf.api.rate, self._monitor, self._configs
        )
        self._vm_controller = VmController(
            self._configs, self._monitor, self._trigger,
            conf.limits.cpu.destroy_on, conf.limits.mem.destroy_on
        )
        self._server = Server(self, port, cpu_power)

        self._destroy_on_stop_service = conf.stop_policy.destroy_on_stop_service
        self._namespace = conf.namespace
        self._active = False
        self._failed_tries = 0

    def run(self):
        self._configs.start()
        self._monitor.start()
        self._reporter.start()
        self._trigger.start()
        self._server.start()
        gevent.sleep(self._sleep_time)

        self._active = True
        while self._active:
            try:
                self._logger.debug('iter_start')
                self._vm_controller.check()
                self._logger.debug('iter_done')
                self._failed_tries = 0
            except Exception as ex:
                self._logger.exception(ex)
                self._failed_tries += 1
                self._logger.warning('%i failed tries', self._failed_tries)
            finally:
                if self._failed_tries > self._wait_till_success:
                    self._logger.error('too many exceptions; exiting')
                    self.stop()
                    sys.exit(32)
                gevent.sleep(self._sleep_time)

    def stop(self):
        self._active = False
        if self._destroy_on_stop_service:
            self._vm_controller.stop(self._namespace)

    @property
    def monitor(self):
        return self._monitor

    @property
    def trigger(self):
        return self._trigger

    @property
    def configs(self):
        return self._configs

    @property
    def events_counter(self):
        return self._vm_controller.events_counter

    def recalc_md5(self):
        self._global_config.recalc_md5()
        self._logger.info('recalced md5')

    def started_ok(self):
        psi = self._monitor.list_psi()
        return (
            (self._trigger.enough_history() and not self.trigger.on) or
            sum(1 for cont in psi if cont.state == 'running')
        )
