import logging
import gevent

from vm.abstract import AbsPsiVM, AbsPsiApp, AbsPsi, ContainerMode
from utils.names import name_to_porto_name, hostname
from utils.online import OnlineState
from utils.configuration import config_to_hash


class CpuLimits(object):
    def __init__(self, min_, max_, null):
        self.min = min_
        self.max = max_
        self.null = null

    def drop(self):
        self.min = self.max = self.null

    def __mul__(self, other):
        assert isinstance(other, (int, float))
        assert 0 <= other <= 1
        new = CpuLimits(max(self.null, self.min * other), max(self.null, self.max * other), self.null)
        return new

    def copy(self):
        return CpuLimits(self.min, self.max, self.null)

    def __str__(self):
        return 'CpuLimits(min=%s, max=%s, null=%s)' % (self.min, self.max, self.null)

    def __repr__(self):
        return str(self)


class MemoryLimits(object):
    def __init__(self, min_, max_, null):
        self.min = min_
        self.max = max_
        self.null = null

    def drop(self):
        self.min = self.max = self.null


class VMConfig(AbsPsiVM):
    def __init__(
        self, namespace, name, dns_name,
        vlans, project_id, layers, root_volume, storages, cluster,
        cpu, mem, net,
        bb_ip, fb_ip,
        porto_props,
    ):
        self._namespace = namespace
        self._name = name
        self._dns_name = dns_name
        self.project_id = project_id
        self.vlans = vlans
        self.bb_ip = bb_ip
        self.fb_ip = fb_ip
        self.layers = layers
        self.root_volume = root_volume
        self.storages = storages
        self.cluster = cluster
        self.net = net

        self.cpu = cpu
        self.mem = mem

        self.porto_props = porto_props

    @property
    def porto_name(self):
        return name_to_porto_name(self._namespace, self._name)

    @property
    def name(self):
        return self._name

    @property
    def dns_name(self):
        return self._dns_name

    @property
    def psi_configuration(self):
        return config_to_hash(
            name=self.name,
            mode=self.mode,
            dns_name=self.dns_name,
            porto_name=self.porto_name,
            root_volume=self.root_volume,
            storages=self.storages,
            cluster=self.cluster,
            layers=self.layers,
            vlans=self.vlans,
            project_id=self.project_id,
            net=self.net,
            bb_ip=self.bb_ip,
            fb_ip=self.fb_ip,
            **self.porto_props
        )


class AppConfig(AbsPsiApp):
    def __init__(
        self, namespace, name, cluster, command,
        cpu, mem, net
    ):
        self._namespace = namespace
        self._name = name
        self._command = command
        self.cluster = cluster
        self.net = net

        self.cpu = cpu
        self.mem = mem

    @property
    def porto_name(self):
        return name_to_porto_name(self._namespace, self._name)

    @property
    def name(self):
        return self._name

    @property
    def command(self):
        return self._command

    @property
    def psi_configuration(self):
        return config_to_hash(
            name=self.name,
            porto_name=self.porto_name,
            command=self.command,
            mode=self.mode
        )


def parse_configs(global_config, response):
    res = {}
    for data in response.json().values():
        cpu = CpuLimits(
            data['limits']['cpu']['min'], data['limits']['cpu']['max'],
            global_config.limits.cpu.null
        )
        mem = MemoryLimits(
            data['limits']['mem']['min'], data['limits']['mem']['max'],
            global_config.limits.mem.null
        )
        conf = data['conf']
        if conf['mode'] == ContainerMode.app:
            c = AppConfig(
                namespace=global_config.namespace,
                name=conf['name'],
                cluster=conf['cluster'],
                command=global_config.command,
                cpu=cpu, mem=mem, net=global_config.limits.net.total,
            )
        elif conf['mode'] == ContainerMode.os:
            c = VMConfig(
                namespace=global_config.namespace,
                name=conf['name'],
                dns_name=conf['dns_name'],
                vlans=global_config.vlans,
                project_id=conf['project_id'],
                layers=global_config.layers[conf['cluster']],
                root_volume=global_config.root_volume,
                storages=global_config.storages,
                cluster=conf['cluster'],
                cpu=cpu, mem=mem, net=global_config.limits.net.total,
                bb_ip=conf.get('bb_ip', None), fb_ip=conf.get('fb_ip', None),
                porto_props=global_config.porto_props,
            )
        res[c.porto_name] = c
    return res


class OnlineConfig(OnlineState):
    _logger = logging.getLogger('configs')
    _hostname = hostname()

    def __init__(self, global_config):
        super(OnlineConfig, self).__init__()
        self._url = global_config.api.url
        self._type = global_config.api.type
        self._sleep_time = global_config.api.rate
        self._stop_after = global_config.stop_policy.stop_after
        self._drop_after = global_config.stop_policy.drop_after
        self._global_config = global_config
        self._containers = {}

    def load_configs(self):
        res = self._get(self.url(
            'configs', self._hostname, compact='1', type=self._type,
            gencfg_ip=('1' if self._global_config.api.gencfg_ip else '0')
        ))
        self._logger.debug('configs loaded, res.ok: {}'.format(res.ok))
        if res.ok:
            self._containers = parse_configs(self._global_config, res)

    def drop_limits(self):
        for cont in self._containers.values():
            cont.cpu.drop()
        self._logger.info('cpu limits dropped')

    def __iter__(self):
        for one in self._containers.values():
            yield one

    def __contains__(self, item):
        if isinstance(item, AbsPsi):
            item = item.porto_name
        return item in self._containers

    def __getitem__(self, item):
        if isinstance(item, AbsPsi):
            item = item.porto_name
        return self._containers[item]

    def _run(self):
        while True:
            try:
                self.load_configs()
            except Exception as ex:
                self._logger.exception(ex)
                self.drop_limits()
            finally:
                if self._stop_after > self.since_upd() > self._drop_after:
                    self.drop_limits()
                elif self.since_upd() > self._stop_after:
                    self._logger.exception('stop all')
                    self._containers = {}
                gevent.sleep(self._sleep_time)
