import os
import retry
import gevent
import logging
import subprocess

from utils import porto_utils as pu
from utils.names import porto_name_to_name, is_psi
from utils.cpu import CpuUsage
from utils.network import vlan_is_up
from vm.abstract import AbsContainer, AbsPsi, AbsPsiVM, AbsPsiApp, ContainerMode


PROBE_INTERVAL = 5
WINDOW_SIZE = 5 * 3


def container_factory(namespace, porto_name):
    if is_psi(namespace, porto_name):
        return psi_factory(namespace, porto_name)
    return Container(porto_name)


def psi_factory(namespace, porto_name):
    mode = pu.get_virt_mode(porto_name)
    if mode == ContainerMode.os:
        return PsiVMContainer(namespace, porto_name)
    if mode == ContainerMode.app:
        return PsiAppContainer(namespace, porto_name)


class Container(AbsContainer):
    def __init__(self, porto_name):
        self._porto_name = porto_name
        self._porto_state = 'none'
        self._porto_mode = None
        self.cpu = {
            'policy': 'normal',
            'usage': CpuUsage(PROBE_INTERVAL, WINDOW_SIZE),
            'max': 0,
            'min': 0
        }
        self.mem = {
            'usage': 0,
            'max': 0,
            'min': 0
        }

    @property
    def porto_name(self):
        return self._porto_name

    @property
    def porto_state(self):
        return self._porto_state

    @property
    def true_mode(self):
        return self._porto_mode

    def is_psi(self):
        return isinstance(self, PsiContainer)

    def is_running(self):
        return self._porto_state == 'running'

    def update_values(self):
        connection = None
        try:
            connection = pu.get_connection()
            self._porto_state = pu.get_state(self.porto_name, connection)
            self._porto_mode = pu.get_virt_mode(self.porto_name, connection)
            if self._porto_state == 'running':
                self.cpu['policy'] = pu.get_cpu_policy(self.porto_name, connection)
                self.cpu['usage'].update(pu.get_cpu_usage(self.porto_name, connection))
                self.cpu['max'] = pu.get_cpu_limit(self.porto_name, connection)
                self.cpu['min'] = pu.get_cpu_guarantee(self.porto_name, connection)

                self.mem['usage'] = pu.get_mem_usage(self.porto_name, connection)
                self.mem['max'] = pu.get_mem_limit(self.porto_name, connection)
                self.mem['min'] = pu.get_mem_guarantee(self.porto_name, connection)
        except pu.PortoBroken:
            self._porto_state = 'dead_porto'
        finally:
            if connection:
                connection.disconnect()


class PsiContainer(Container, AbsPsi):
    def __init__(self, psi_namespace, porto_name):
        super(PsiContainer, self).__init__(porto_name)
        self._namespace = psi_namespace
        self._psi_configuration = None
        self._cluster = None

        self.io = {
            'read': 0,
            'write': 0,
        }
        self.net = {
            'limit': 0,
            'rx': 0,
            'tx': 0,
        }

    @property
    def name(self):
        return porto_name_to_name(self._namespace, self.porto_name)

    @property
    def psi_configuration(self):
        return self._psi_configuration

    @property
    def cluster(self):
        return self._cluster

    @property
    def state(self):
        return self._porto_state

    def is_active(self):
        return self.state == 'running'

    @staticmethod
    def _get_psi_info(env):
        res = dict(psi_configuration='', cluster='')
        for k_v in env.split(';'):
            k_v = k_v.strip()
            if k_v.startswith('PSI_CONFIGURATION'):
                res['psi_configuration'] = k_v.split('=')[1]
            if k_v.startswith('CLUSTER'):
                res['cluster'] = k_v.split('=')[1]
        return res

    def update_values(self):
        super(PsiContainer, self).update_values()
        try:
            info = self._get_psi_info(pu.get_env(self.porto_name))
            self._psi_configuration = info['psi_configuration']
            self._cluster = info['cluster']

            if not self.is_running():
                return
            with pu.connect() as connection:
                self.net['limit'] = pu.get_total_net_limit(self._porto_name, connection)
                self.net['rx'] = pu.get_total_net_rx_bytes(self._porto_name, connection)
                self.net['tx'] = pu.get_total_net_tx_bytes(self._porto_name, connection)
                self.io['read'] = pu.get_total_io_read(self._porto_name, connection)
                self.io['write'] = pu.get_total_io_write(self._porto_name, connection)
        except pu.PortoBroken:
            self._porto_state = 'dead_porto'

    def _json(self):
        return {
            'limits': {
                'cpu': {
                    'max': self.cpu['max'], 'min': self.cpu['min']
                },
                'mem': {
                    'max': self.mem['max'], 'min': self.mem['min']
                },
                'net': {
                    'total': self.net['limit']
                }
            },
            'usage': {
                'cpu': self.cpu['usage'].value,
                'mem': self.mem['usage']
            }
        }


class PsiVMContainer(PsiContainer, AbsPsiVM):
    def __init__(self, namespace, porto_name):
        super(PsiVMContainer, self).__init__(namespace, porto_name)
        self._dns_name = None
        self._state = 'none'

    @property
    def dns_name(self):
        return self._dns_name

    @property
    def state(self):
        return self._state

    def check_skynet(self):
        pop = subprocess.Popen([
            'sky', 'ping', '--extended', '-t', '2', self.dns_name],
            stdout=subprocess.PIPE, stderr=subprocess.PIPE
        )
        pop.communicate()
        if pop.returncode != 0:
            return 'skynet_broken'
        return None

    def ping_porto(self):
        pop = subprocess.Popen(['ping6', '-c', '1', '-w', '2', self.dns_name],
                               stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        res = pop.communicate()
        if pop.returncode != 0:
            if 'unknown host' in res[1]:
                return 'dns_problem'
            else:
                return 'network_broken'
        return None

    def check_volume(self):
        path = pu.get_root(self.porto_name) + '/sbin'
        if not os.path.exists(path):
            return 'volume_broken'
        return None

    def update_values(self):
        super(PsiVMContainer, self).update_values()
        try:
            self._dns_name = pu.get_hostname(self.porto_name)
        except pu.PortoBroken:
            self._porto_state = 'dead_porto'
        if self._porto_state == 'running':
            self._state = self.check_volume() or self.ping_porto() or self._porto_state
        else:
            self._state = self.check_volume() or self._porto_state

    @classmethod
    def default_json(cls, dns_name, name, vlans):
        all_vlans = all([vlan_is_up(vlan) for vlan in vlans.values()])
        state = 'not_listed' if all_vlans else 'no_vlans'
        res = {
            'state': state,
            'dns_name': dns_name,
            'name': name,
            'active': False,
            'limits': None,
            'usage': None,
            'mode': ContainerMode.os
        }
        return res

    def json(self):
        res = {
            'state': self.state,
            'dns_name': self.dns_name,
            'name': self.name,
            'active': self.is_active(),
            'limits': None,
            'usage': None,
            'vlans': None,
            'mode': ContainerMode.os
        }
        if self.porto_state == 'running':
            res.update(self._json())
        return res


class PsiAppContainer(PsiContainer, AbsPsiApp):
    def __init__(self, namespace, porto_name):
        super(PsiAppContainer, self).__init__(namespace, porto_name)
        self._command = None

    @property
    def command(self):
        return self._command

    def update_values(self):
        super(PsiAppContainer, self).update_values()
        try:
            self._command = pu.get_command(self.porto_name)
        except pu.PortoBroken:
            self._porto_state = 'dead_porto'

    @classmethod
    def default_json(cls, name):
        res = {
            'state': 'not_listed',
            'name': name,
            'active': False,
            'limits': None,
            'usage': None,
            'mode': ContainerMode.app
        }
        return res

    def json(self):
        res = {
            'state': self.state,
            'name': self.name,
            'active': self.is_active(),
            'limits': None,
            'usage': None,
            'mode': ContainerMode.app
        }
        if self.porto_state == 'running':
            res.update(self._json())
        return res


class Monitor(gevent.Greenlet):
    sleep_time = PROBE_INTERVAL
    _logger = logging.getLogger('Monitor')

    def __init__(self, psi_namespace, vlans, window_size):
        super(Monitor, self).__init__()
        self._namespace = psi_namespace
        self._vlans = vlans
        self._containers = {}

        # tiny hack
        global WINDOW_SIZE
        WINDOW_SIZE = window_size

    def _check_psi_mode(self):
        for one in self.list_psi():
            if one.mode != one.true_mode:
                del self._containers[one.porto_name]

    @retry.retry(pu.PortoBaseError, 3, 1)
    def _check(self):
        names = set()
        for porto_name in pu.list_containers('{}/*'.format(self._namespace)):
            if porto_name not in self._containers:
                self._containers[porto_name] = container_factory(self._namespace, porto_name)
            self._containers[porto_name].update_values()
            names.add(porto_name)
        for porto_name in self._containers.keys():
            if porto_name not in names:
                del self._containers[porto_name]

    def _run(self):
        while True:
            try:
                self._check()
            except Exception as ex:
                self._logger.exception(ex)
            finally:
                gevent.sleep(self.sleep_time)

    def filter(self, key_func):
        return [one for one in self._containers.values() if key_func(one)]

    def list_psi(self):
        return self.filter(Container.is_psi)

    def default_json(self, container):
        if isinstance(container, AbsPsiVM):
            return PsiVMContainer.default_json(container.dns_name, container.name, self._vlans)
        if isinstance(container, AbsPsiApp):
            return PsiAppContainer.default_json(container.name)

    def __getitem__(self, item):
        if isinstance(item, AbsPsi):
            item = item.porto_name
        return self._containers[item]

    def __contains__(self, item):
        if isinstance(item, AbsPsi):
            item = item.porto_name
        return item in self._containers

    def __iter__(self):
        for one in self._containers.values():
            yield one


__all__ = ['Monitor']
