import logging
import gevent
import gevent.pywsgi
import flask
import json
from utils.cpu import cpu_total, cpu_count


SIG_OPT = '_ammv'
SIG_OPT_DELTA = '_dmmv'
SIG_OPT_SUM = '_summ'


class YasmSignals(object):
    def __init__(self, monitor, trigger, configs, events_counter, cpu_power):
        self.monitor = monitor
        self.trigger = trigger
        self.configs = configs
        self.events_counter = events_counter
        self.cpu_power = cpu_power

    def all_signals(self):
        return self.host_signals() + self.psi_signals() + self.events_signals()

    def host_signals(self):
        return [
            self.host_cpu_usage(),
            self.host_configs_last_upd()
        ]

    def psi_signals(self):
        res = []
        for cluster in self._all_clusters():
            if self._filter_cluster(cluster):
                res += [
                    self.psi_cpu_usage(cluster),
                    self.psi_cpu_limit(cluster),
                    self.psi_mem_usage(cluster),
                    self.psi_mem_limit(cluster),
                    self.psi_io_read(cluster),
                    self.psi_io_write(cluster),
                    self.psi_net_limit(cluster),
                    self.psi_net_tx_bytes(cluster),
                    self.psi_net_rx_bytes(cluster),
                    self.psi_containers_cnt(cluster),
                ]
        return res

    def events_signals(self):
        return [
            [
                'prj={cluster};event_{event}{sig_opt}'.format(cluster=cluster, event=event, sig_opt=SIG_OPT_SUM),
                value
            ]
            for cluster, event, value in self.events_counter.to_tuples()
        ]

    def host_cpu_usage(self):
        return [
            'host_cpu_usage' + SIG_OPT,
            self.cpu_to_power(self.trigger.value * cpu_count())
        ]

    def host_configs_last_upd(self):
        return [
            'host_configs_last_upd' + '_xhhh',
            self.configs.since_upd(True) or 0
        ]

    def _filter_cluster(self, cluster):
        return [one for one in self.monitor.list_psi() if one.cluster == cluster]

    def _all_clusters(self):
        return {one.cluster for one in self.monitor.list_psi() if one.cluster}

    def psi_cpu_usage(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_cpu_usage' + SIG_OPT,
            self.cpu_to_power(sum(one.cpu['usage'].value for one in self._filter_cluster(cluster)))
        ]

    def psi_mem_usage(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_mem_usage' + SIG_OPT,
            sum(one.mem['usage'] for one in self._filter_cluster(cluster))
        ]

    def psi_cpu_limit(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_cpu_limit' + SIG_OPT,
            self.cpu_to_power(sum(one.cpu['max'] for one in self._filter_cluster(cluster)))
        ]

    def psi_mem_limit(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_mem_limit' + SIG_OPT,
            sum(one.mem['max'] for one in self._filter_cluster(cluster))
        ]

    def psi_containers_cnt(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_containers_cnt' + SIG_OPT,
            len(self._filter_cluster(cluster))
        ]

    def psi_io_read(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_io_read' + SIG_OPT_DELTA,
            sum(one.io['read'] for one in self._filter_cluster(cluster))
        ]

    def psi_io_write(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_io_write' + SIG_OPT_DELTA,
            sum(one.io['write'] for one in self._filter_cluster(cluster))
        ]

    def psi_net_limit(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_net_limit' + SIG_OPT,
            sum(one.net['limit'] for one in self._filter_cluster(cluster))
        ]

    def psi_net_tx_bytes(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_net_tx_bytes' + SIG_OPT_DELTA,
            sum(one.net['tx'] for one in self._filter_cluster(cluster))
        ]

    def psi_net_rx_bytes(self, cluster):
        return [
            'prj={};'.format(cluster) + 'psi_net_rx_bytes' + SIG_OPT_DELTA,
            sum(one.net['rx'] for one in self._filter_cluster(cluster))
        ]

    def cpu_to_power(self, cpu):
        return float(self.cpu_power) * cpu / cpu_total()


class Server(gevent.Greenlet):
    def __init__(self, service, port, cpu_power):
        super(Server, self).__init__()
        self.service = service
        self.port = port
        self.yasm = YasmSignals(service.monitor, service.trigger, service.configs, service.events_counter, cpu_power)

    def golovan_metrics(self):
        return flask.Response(json.dumps(self.yasm.all_signals()))

    def started_ok(self):
        return flask.jsonify({'ok': self.service.started_ok()})

    def recalc_md5(self):
        try:
            _log.info('got recalc_md5 request')
            self.service.recalc_md5()
            return flask.jsonify({'ok': True})
        except Exception as ex:
            _log.exception(ex)
            return flask.jsonify({'ok': False})

    def stop_service(self):
        try:
            _log.info('got stop_service request')
            self.service.stop()
            return flask.jsonify({'ok': True})
        except Exception as ex:
            _log.exception(ex)
            return flask.jsonify({'ok': False})

    def _run(self):
        app = flask.Flask(__name__)
        app.add_url_rule('/yasm/signals', 'yasm', view_func=self.golovan_metrics)
        app.add_url_rule('/started_ok', 'started_ok', view_func=self.started_ok)
        app.add_url_rule('/recalc_md5', 'recalc_md5', view_func=self.recalc_md5, methods=['POST'])
        app.add_url_rule('/stop_service', 'stop_service', view_func=self.stop_service, methods=['POST'])
        gevent.pywsgi.WSGIServer(('::', int(self.port)), app).serve_forever()


_log = logging.getLogger('server')
