import math
import gevent
import psutil
import logging
from utils.cpu import MovingAverage


class Trigger(gevent.Greenlet):
    _logger = logging.getLogger('trigger')

    def __init__(self, interval, window_size, low_watermark, high_watermark, probe=psutil.cpu_percent):
        super(Trigger, self).__init__()

        self.probe = probe
        self.probe_interval = float(interval)
        self.window_size = float(window_size)
        self.high_watermark = float(high_watermark)
        self.low_watermark = float(low_watermark)
        self.time_elapsed = 0

        self.avg = MovingAverage(1 - math.exp(-self.probe_interval/self.window_size))

    @property
    def on(self):
        return self.enough_history() and self.value < self.low_watermark

    @property
    def off(self):
        return self.enough_history() and self.value > self.high_watermark

    @property
    def value(self):
        return self.avg.value

    def _run(self):
        while True:
            data = self.probe()
            self.avg.push(data)
            self._logger.debug('%s %s', data, self.avg.value)

            gevent.sleep(self.probe_interval)

            self.time_elapsed += self.probe_interval

    def enough_history(self):
        return self.time_elapsed > self.window_size
