import os
import abc
import porto
import socket
import gevent
import logging
import subprocess
import porto_utils as pu


def sleep(secs):
    gevent.sleep(secs)


def filter_none(**kwargs):
    return {k: v for k, v in kwargs.items() if v is not None}


def unlink_others(volume, our_name):
    for cont in volume.GetContainers():
        if cont.name != our_name:
            volume.Unlink(cont.name)


def get_active_interface(iface_regex):
    iface_up = ''
    ifaces = os.listdir('/sys/class/net/')
    if ifaces:
        for iface in ifaces:
            if iface_regex in iface:
                if 'up' in open(os.path.join('/sys/class/net', iface, 'operstate')).read():
                    iface_up = iface
        if iface_up:
            return iface_up
        else:
            raise Exception("Not found active interfaces in /sys/class/net/")
    else:
        raise Exception("Not found interfaces in /sys/class/net/")


def generate_ip_from_prj_id(project_id, iface, uniq):
    ipaddr_dot_list = []
    with open('/proc/net/if_inet6') as if_inet6:
        for if_inet6_s in if_inet6:
            if_inet6_s = if_inet6_s.split(' ')
            if if_inet6_s[-1].strip() == iface and if_inet6_s[3] == '00':
                ipaddr_cur_str = if_inet6_s[0]
                ipaddr_str = str(ipaddr_cur_str[:16]) + str(project_id).rjust(8, '0') + str(int(uniq, base=16)).rjust(8, '0')
                ipaddr_dot = ':'.join(map(''.join, (zip(*[iter(ipaddr_str)]*4))))
                ipaddr_dot = socket.inet_ntop(socket.AF_INET6, socket.inet_pton(socket.AF_INET6, ipaddr_dot))
                ipaddr_dot_list.append(ipaddr_dot)
    if ipaddr_dot_list:
        return ipaddr_dot_list
    else:
        raise Exception("Error generate IP for PROJECT_ID: " + project_id)


class Config(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def config(self):
        pass


class Resource(Config):
    @abc.abstractmethod
    def prepare(self, *args, **kwargs):
        pass


class NetConfigs(Config):
    def get_bb_up(self):
        return get_active_interface(self.iface_bb)

    def get_bb(self):
        return "macvlan {0} {1}".format(self.get_bb_up(), 'eth0')

    def get_fb_up(self):
        return get_active_interface(self.iface_fb)

    def get_fb(self):
        return "macvlan {0} {1}".format(self.get_fb_up(), self.get_fb_up())

    def get_hbf_up(self):
        return get_active_interface(self.iface_hbf)

    def get_hbf(self):
        return "macvlan {0} {1}".format(self.get_hbf_up(), self.get_hbf_up())

    def __init__(
        self, iface_fb, iface_bb, iface_hbf,
        net_limit_fb, net_limit_bb, net_limit_hbf
    ):
        self.iface_fb = iface_fb
        self.iface_bb = iface_bb
        self.iface_hbf = iface_hbf
        self.net_limit_fb = net_limit_fb
        self.net_limit_bb = net_limit_bb
        self.net_limit_hbf = net_limit_hbf

    def config(self):
        return {
            'net': ';'.join([self.get_bb(), self.get_fb(), self.get_hbf()]),
            'net_limit[eth0]': self.net_limit_bb,
            'net_limit[{}]'.format(self.get_fb_up()): self.net_limit_fb,
            'net_limit[{}]'.format(self.get_hbf_up()): self.net_limit_hbf,
        }


class NetConfigs64(Config):
    def get_bb_up(self):
        return get_active_interface(self.iface_bb)

    def get_fb_up(self):
        return get_active_interface(self.iface_fb)

    def get_fb_ip(self):
        if self.fb_ip:
            return [self.fb_ip]
        return generate_ip_from_prj_id(self.project_id, self.get_fb_up(), str(self.port))

    def get_bb_ip(self):
        if self.bb_ip:
            return [self.bb_ip]
        return generate_ip_from_prj_id(self.project_id, self.get_bb_up(), str(self.port))

    def __init__(
        self, iface_fb, iface_bb, net_limit, bb_ip=None, fb_ip=None, project_id=None, port=None
    ):
        self.iface_fb = iface_fb
        self.iface_bb = iface_bb
        self.net_limit = net_limit
        self.bb_ip = bb_ip
        self.fb_ip = fb_ip
        self.project_id = project_id
        self.port = port

    def config(self):
        return {
            'net': 'L3 {}'.format(self.get_bb_up()),
            'ip': ';'.join(['{} {}'.format(self.get_bb_up(), ip) for ip in self.get_bb_ip() + self.get_fb_ip() if ip]),
            'net_limit[eth0]': self.net_limit
        }


class DefaultNetConfigs(Config):
    def __init__(self, net_limit):
        self.net_limit = net_limit

    def config(self):
        return {
            'net_limit[default]': self.net_limit
        }


class LimitsConfig(Config):
    def __init__(self, mem_limit, cpu_limit):
        self.mem_limit = mem_limit
        self.cpu_limit = cpu_limit

    def config(self):
        return {
            'memory_limit': self.mem_limit,
            'memory_guarantee': self.mem_limit,
            'cpu_limit': pu.percent_to_cores(self.cpu_limit),
            'cpu_guarantee': pu.percent_to_cores(self.cpu_limit),
        }


class Capabilities(object):
    def __init__(self, allow_mount=False):
        self.allow_mount = allow_mount

    def list(self):
        lst = []
        if self.allow_mount:
            lst.append('CAP_SYS_ADMIN')
        return lst


class BindStorageResource(Resource):
    sleep_time = 1

    def __init__(self, fullname, host_path, vm_path, space_limit):
        self.fullname = bytes(fullname)
        self.host_path = bytes(host_path)
        self.vm_path = bytes(vm_path)
        self.space_limit = bytes(space_limit)

        self._conn = pu.get_connection()
        self._volume = None

    @property
    def conn(self):
        return self._conn

    def chmod(self, value='777'):
        subprocess.Popen(
            ['chmod', value, self.host_path],
        ).communicate()

    def prepare(self):
        if self._volume is None:
            self._volume = self.conn.CreateVolume(space_limit=self.space_limit, storage=self.host_path)
            sleep(self.sleep_time)
            self._volume.Link(self.fullname)
            sleep(self.sleep_time)
            unlink_others(self._volume, self.fullname)
            sleep(self.sleep_time)
            self.chmod()

    def config(self):
        return {
            'bind': '{} {}'.format(self._volume.path, self.vm_path)
        }


class VolumeStorageResource(Resource):
    sleep_time = 1

    def __init__(self, fullname, host_path, vm_path, space_limit, volume_directory):
        self.fullname = bytes(fullname)
        self.host_path = bytes(host_path)
        self.vm_path = bytes(vm_path.lstrip('/'))
        self.space_limit = space_limit
        self.volume_directory = bytes(volume_directory)

        self._conn = pu.get_connection()
        self._volume = None

    @property
    def conn(self):
        return self._conn

    @staticmethod
    def chmod(path, value='777'):
        subprocess.Popen(
            ['chmod', value, path]
        ).communicate()

    def make_dirs(self, root, path):
        logging.info('%s %s', root, path)
        self.chmod(root)
        temp = root
        for part in path.split('/'):
            temp = os.path.join(temp, part)
            if not os.path.exists(temp):
                logging.info('makedirs %s', temp)
                os.makedirs(temp)
            self.chmod(temp)
            logging.info('chmod %s', temp)

    def prepare(self):
        mount_path = os.path.join(self.volume_directory, self.vm_path)
        if self._volume is None:
            self.make_dirs(self.volume_directory, self.vm_path)
            self._volume = self.conn.CreateVolume(
                space_limit=self.space_limit, storage=self.host_path,
                path=mount_path
            )
            sleep(self.sleep_time)
            self._volume.Link(self.fullname)
            sleep(self.sleep_time)
            unlink_others(self._volume, self.fullname)
            sleep(self.sleep_time)

    def config(self):
        return {}


class VolumeResource(Resource):
    sleep_time = 2

    def __init__(self, fullname, volume_directory, layers, space_limit, psi_configuration, storage_path=None):
        self.volume_directory = bytes(volume_directory)
        self.space_limit = bytes(space_limit)
        self.layers = [bytes(layer) for layer in layers]
        self.fullname = bytes(fullname)
        self.psi_configuration = bytes(psi_configuration)
        self.storage_path = storage_path

        self._conn = pu.get_connection(90)

    @property
    def conn(self):
        return self._conn

    def format_layer_name(self, layer):
        return '_weak_{}-{}-{}'.format(
            self.fullname.replace("/", "_"), self.psi_configuration, self.layers.index(layer)
        )

    def prepare(self):
        layers_names = []
        for layer in self.layers:
            layer_name = self.format_layer_name(layer)
            try:
                self.conn.FindLayer(layer_name)
            except porto.exceptions.LayerNotFound:
                self.conn.ImportLayer(layer_name, layer)
            layers_names.append(layer_name)
            sleep(self.sleep_time)
        try:
            volume = self.conn.FindVolume(path=self.volume_directory)
        except porto.exceptions.VolumeNotFound:
            volume = self.conn.CreateVolume(**filter_none(
                path=self.volume_directory, layers=layers_names,
                space_limit=self.space_limit,
                storage=self.storage_path,
            ))
        sleep(self.sleep_time)
        try:
            volume.Link(self.fullname)
        except porto.exceptions.VolumeAlreadyLinked:
            pass
        sleep(self.sleep_time)
        unlink_others(volume, self.fullname)
        sleep(self.sleep_time)

    def config(self):
        return {}


class VMContainerConfig(object):
    def __init__(self, fullname, dns_name, root, porto_props):
        self.fullname = bytes(fullname)
        self.dns_name = bytes(dns_name)
        self.root = bytes(root)
        self.porto_props = porto_props

    @property
    def mode(self):
        return 'os'

    def config(self):
        conf = {
            'virt_mode': 'os',
            'command': '/sbin/init',
            'porto_namespace': '{}/'.format(self.fullname),
            'hostname': self.dns_name,
            'root': self.root,
        }
        conf.update(self.porto_props)
        return conf


class AppContainerConfig(object):
    def __init__(self, fullname, command, auto_namespace):
        self.fullname = bytes(fullname)
        self.command = bytes(command)
        self.auto_namespace = auto_namespace

    @property
    def mode(self):
        return 'app'

    def config(self):
        config = {
            'virt_mode': 'app',
            'command': self.command,
            'porto_namespace': '{}/'.format(self.fullname)
        }
        if self.auto_namespace:
            del config['porto_namespace']
            logging.warn('porto namespace is set to None')
        return config


class Launcher(object):
    sleep_time = 2

    def __init__(
        self, container_config, net_config, limits_config,
        meta_info, volume=None, storages=None, capabilities=None,
    ):
        self.container_config = container_config
        self.net_config = net_config
        self.limits_config = limits_config
        self.volume_resource = volume
        self.storages_resources = storages or []

        self.capabilities = capabilities
        self.meta_info = meta_info

        self._conn = pu.get_connection(90)
        self._logger = logging.getLogger('Launcher')

    @property
    def conn(self):
        return self._conn

    @property
    def fullname(self):
        return self.container_config.fullname

    def create_container(self):
        try:
            self.conn.Create(self.fullname)
        except porto.exceptions.ContainerAlreadyExists:
            self.conn.Destroy(self.fullname)
            sleep(self.sleep_time)
            self.conn.Create(self.fullname)
        self.conn.SetProperty(self.fullname, 'virt_mode', self.container_config.mode)
        sleep(self.sleep_time)

    def set_property(self, key, value):
        self.conn.SetProperty(self.fullname, key, value)

    def get_property(self, key):
        return self.conn.GetProperty(self.fullname, key)

    def set_properties(self):
        self._logger.debug(self.config())
        for (key, value) in self.config().items():
            self.set_property(key, value)

    def set_capabilities(self):
        if not self.capabilities:
            return
        caps = self.get_property('capabilities').split(';')
        caps += self.capabilities.list()
        caps = ';'.join(caps)
        self.set_property('capabilities', caps)

    def prepare_storages(self):
        for storage in self.storages_resources:
            storage.prepare()

    @staticmethod
    def default_config():
        return {
            'root_readonly': 'false',
            'recharge_on_pgfault': 'true',
            'cpu_policy': 'normal',
            'stdout_path': '/dev/null',
            'stderr_path': '/dev/null',
            'stdin_path': '/dev/null',
        }

    def config(self):
        conf = self.default_config()
        conf.update(self.container_config.config())
        conf.update(self.limits_config.config())
        conf.update(self.net_config.config())
        if self.volume_resource:
            conf.update(self.volume_resource.config())
        if self.storages_resources:
            bind_ = ';'.join(
                storage.config()['bind']
                for storage in self.storages_resources
                if isinstance(storage, BindStorageResource)
            )
            conf.update(bind=bind_)
        return conf

    def get_env(self):
        params = [
            key.upper().replace('[', '_').replace(']', '_') + '=' + str(value)
            for key, value in self.config().items()
            if key not in {'command'}
        ]
        for key, value in self.meta_info.items():
            params.append('{}={}'.format(str(key).upper(), str(value)))
        return '; '.join(params)

    def set_env(self):
        self._logger.debug(self.get_env())
        self.set_property('env', self.get_env())

    def prepare(self):
        self.create_container()
        self._logger.info('created container')
        sleep(self.sleep_time)
        if self.volume_resource:
            self.volume_resource.prepare()
            self._logger.info('prepared volume')
        sleep(self.sleep_time)
        if self.storages_resources:
            self.prepare_storages()
            self._logger.info('prepared storages')
            sleep(self.sleep_time)
        self.set_properties()
        self._logger.info('set properties')
        self.set_capabilities()
        self._logger.info('set capabilities')
        sleep(self.sleep_time)
        self.set_env()
        self._logger.info('set env')
        sleep(self.sleep_time)

    def start(self):
        self.conn.Start(self.fullname)
        self._logger.info('started container')
        sleep(self.sleep_time)
