import contextlib
import logging
import socket

import gevent

from os_utils import execute


def setup_vlan(vlan):
    vlan = str(vlan)
    link = 'vlan{}'.format(vlan)

    if _link_is_up(link):
        _log.debug('vlan %s is already up', vlan)
    else:
        eth = _get_eth()
        _add_link(eth, link, vlan)
        _up_link(link)

        if not _link_is_up(link):
            raise RuntimeError('Unable to setup vlan {}'.format(vlan))

        _log.debug('vlan %s successfully set up', vlan)


def read_tcp(host, port):
    with gevent.Timeout(seconds=5, exception=IOError):
        with contextlib.closing(socket.socket(socket.AF_INET6, socket.SOCK_STREAM)) as sock:
            sock.connect((host, port))
            sock.send('')
            return sock.recv(8000)


def _get_eth():
    for line in execute(['/sbin/ip', 'link'], collect_stdout=True)[1].splitlines():
        if 'vlan' in line:
            return line.split()[1].split('@')[1].replace(':', '')

    raise RuntimeError('Unable to determine network interface')


def _link_is_up(link):
    return 'UP' in execute(['/sbin/ip', 'link', 'show', link], collect_stdout=True)[1]


def vlan_is_up(vlan):
    vlan = str(vlan)
    link = 'vlan{}'.format(vlan)
    return _link_is_up(link)


def _add_link(eth, link, vlan):
    execute(['sudo', '/sbin/ip', 'link', 'add', 'link', eth, 'name', link, 'type', 'vlan', 'id', vlan])


def _up_link(link):
    execute(['sudo', '/sbin/ip', 'link', 'set', link, 'up'])


_log = logging.getLogger(__name__)
