import gevent
import gevent.select
import logging
import os
import select
import shutil
import subprocess
import time


def execute(popenargs, raise_on_error=False, collect_stdout=False, collect_stderr=False, debug=False, **kwargs):
    logger = logging.getLogger('subprocess')

    stdout = ''
    stderr = ''
    child = subprocess.Popen(popenargs, stdout=subprocess.PIPE, stderr=subprocess.PIPE, **kwargs)
    if debug:
        logger.info('[%s] call "%s"', child.pid, popenargs)

    p = gevent.select.poll()
    p.register(child.stdout, select.POLLIN)
    p.register(child.stderr, select.POLLIN)

    events = p.poll()
    while True:
        for rfd, event in events:
            if event & select.POLLIN:
                if rfd == child.stdout.fileno():
                    line = child.stdout.readline()
                    if line:
                        if debug:
                            logger.debug('[%s] %s', child.pid, line[:-1])
                        if collect_stdout:
                            stdout += line
                    else:
                        p.unregister(child.stdout)

                elif rfd == child.stderr.fileno():
                    line = child.stderr.readline()
                    if line:
                        logger.error('[%s] %s', child.pid, line[:-1])
                        if collect_stderr:
                            stderr += line
                    else:
                        p.unregister(child.stderr)

        if not p.fds:
            break

        events = p.poll()

    retcode = child.wait()

    if debug:
        logger.info('[%s] call returned %s', child.pid, retcode)

    if retcode and raise_on_error:
        raise RuntimeError('Subprocess failed with code {}'.format(retcode))

    return retcode, stdout, stderr


def recreate_dir(path):
    remove_dir(path)
    os.makedirs(path)


def remove_dir(path):
    if os.path.exists(path):
        shutil.rmtree(path)


def ensure_dir(path):
    if not os.path.exists(path):
        os.makedirs(path)


def chmod_dir(path, mode='777'):
    std = subprocess.Popen(['chmod', mode, path], stdout=subprocess.PIPE, stderr=subprocess.PIPE).communicate()
    logging.info(std)


def create_symlink(src, dst):
    os.symlink(src, dst)


def remove_file(path):
    try:
        os.remove(path)
    except OSError:
        pass


def sky_get(files, cwd):
    popens = []
    for resource in files:
        popens.append(subprocess.Popen(['sky', 'get', '-uw', resource['resid']], stdout=subprocess.PIPE, cwd=cwd))
    for p in popens:
        p.communicate()
        if p.returncode != 0:
            raise RuntimeError()
    return


def ls(path):
    return [os.path.join(path, d) for d in os.listdir(path)]


def free_space(path):
    s = os.statvfs(path)
    return s.f_frsize * s.f_bavail


def du(path, exclude_dotfiles=True):
    p = subprocess.Popen(
        'du -bs {} {}'.format('--exclude ".tmp"' if exclude_dotfiles else '', path),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        shell=True,
    )
    out, err = p.communicate()

    if p.returncode:
        raise RuntimeError('du failed with code {}, stderr: [{}]'.format(p.returncode, err))

    return int(out.split()[0])


def loop_forever(fn, *args, **kwargs):
    while True:
        # noinspection PyBroadException
        try:
            fn(*args, **kwargs)
        except Exception:
            logging.exception('Unhandled exception in the loop <%s>', fn)

        gevent.sleep(1)


def retry_with_default_value(exc=Exception, tries=3, delay=1, default=None):
    def decorator(func):
        def wrapped(*args, **kwargs):
            for i in xrange(tries):
                try:
                    res = func(*args, **kwargs)
                    return res
                except exc:
                    time.sleep(delay)
            return default
        return wrapped
    return decorator


def test_execute():
    from gevent import pool
    p = pool.Pool(10)

    def f1(i):
        execute(['echo 42; sleep 1; echo 43'], shell=True)
        return i

    for r in p.imap(f1, xrange(10)):
        print r

if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG)
    test_execute()
