import logging
import os

from utils import porto_utils as pu
from utils import os_utils, launcher2
from utils import names


class LimitProblem(RuntimeError):
    pass


def ensure_destroyed(porto_name):
    try:
        pu.destroy_container(porto_name)
        _log.info('destroyed container')
    except pu.ContainerDoesNotExist:
        return


def apply_cpu(porto_name, cpu_min, cpu_max):
    if not is_started(porto_name):
        return
    _log.info('%s apply cpu --> %.2f, %.2f', porto_name, cpu_min, cpu_max)
    try:
        pu.set_cpu_guarantee(porto_name, cpu_min)
        pu.set_cpu_limit(porto_name, cpu_max)
    except pu.LimitProblems:
        raise LimitProblem()


def apply_mem(porto_name, mem_min, mem_max):
    if not is_started(porto_name):
        return
    _log.info('%s apply mem --> %2.f, %2.f', porto_name, mem_min, mem_max)
    try:
        pu.set_mem_guarantee(porto_name, str(mem_min))
        pu.set_mem_limit(porto_name, str(mem_max))
    except pu.LimitProblems:
        raise LimitProblem()


def is_started(porto_name):
    try:
        pu.find_container(porto_name)
    except pu.ContainerDoesNotExist:
        return False
    except pu.PortoBaseError:
        return None
    return True


def is_dead(porto_name):
    try:
        return pu.get_state(porto_name) in {'dead', 'stopped'}
    except pu.PortoBaseError:
        return None


def launch_vm(config, cpu_start, mem_start):
    _log.info('launching %s', config.porto_name)
    _prepare_vm(config)
    _launch_vm(config, cpu_start, mem_start)
    _log.info('done launching %s', config.porto_name)


def _prepare_vm(config):
    def _prepare_dir(directory):
        os_utils.ensure_dir(directory)
        os_utils.chmod_dir(directory)

    pu.null_on_raise(pu.destroy_container, config.porto_name)
    pu.ensure_container_path(config.porto_name)

    _prepare_dir(_volume_directory(config))
    _prepare_dir(_layers_directory(config))

    for storage_conf in config.storages:
        if not storage_conf.clusters or config.cluster in storage_conf.clusters:
            _prepare_dir(_storage_directory(config, storage_conf))
    if config.root_volume.storage_path:
        _prepare_dir(_root_storage_directory(config))


def _launch_vm(config, cpu_start, mem_start):
    vlans = {
        key: 'vlan' + vlan for key, vlan in config.vlans.items()
    }
    if config.fb_ip:
        net_config = launcher2.NetConfigs64(
            vlans['fb'], vlans['bb'], config.net, config.bb_ip, config.fb_ip,
            None, None
        )
    else:
        net_config = launcher2.NetConfigs64(
            vlans['fb'], vlans['bb'], config.net, None, None,
            config.project_id, names.dns_to_port(config.dns_name)
        )
    limits_config = launcher2.LimitsConfig(mem_start, cpu_start)
    container_config = launcher2.VMContainerConfig(
        config.porto_name, config.dns_name, _volume_directory(config),
        config.porto_props,
    )
    root_storage_path = _root_storage_directory(config) if config.root_volume.storage_path else None
    volume = launcher2.VolumeResource(
        fullname=config.porto_name,
        volume_directory=_volume_directory(config),
        layers=[os.path.join(_layers_directory(config), layer['name']) for layer in config.layers],
        space_limit=config.root_volume.space,
        psi_configuration=config.psi_configuration,
        storage_path=root_storage_path,
    )

    storages = []
    for storage_conf in config.storages:
        storage_config = launcher2.BindStorageResource(
            config.porto_name, _storage_directory(config, storage_conf), storage_conf.vm, storage_conf.space
        )
        if not storage_conf.clusters or config.cluster in storage_conf.clusters:
            storages.append(storage_config)

    launcher = launcher2.Launcher(
        container_config=container_config,
        volume=volume, storages=storages,
        limits_config=limits_config,
        net_config=net_config,
        meta_info=dict(psi_configuration=config.psi_configuration, cluster=config.cluster)
    )
    launcher.prepare()
    launcher.start()


def launch_app(config, cpu_start, mem_start):
    _log.info('launching %s', config.porto_name)
    _prepare_app(config)
    _launch_app(config, cpu_start, mem_start)
    _log.info('done launching %s', config.porto_name)


def _prepare_app(config):
    pu.null_on_raise(pu.destroy_container, config.porto_name)
    pu.ensure_container_path(config.porto_name)


def _launch_app(config, cpu_start, mem_start):
    container_config = launcher2.AppContainerConfig(
        config.porto_name, config.command, auto_namespace=False
    )
    net_config = launcher2.DefaultNetConfigs(config.net)
    limits_config = launcher2.LimitsConfig(mem_start, cpu_start)
    launcher = launcher2.Launcher(
        container_config=container_config, net_config=net_config, limits_config=limits_config,
        meta_info=dict(psi_configuration=config.psi_configuration, cluster=config.cluster)
    )
    launcher.prepare()
    launcher.start()


def _volume_directory(config):
    return os.path.join(config.root_volume.volume_path, config.name, 'volume')


def _root_storage_directory(config):
    return os.path.join(config.root_volume.storage_path, config.name, 'root')


def _storage_directory(config, storage_config):
    return os.path.join(storage_config.host, config.name, 'storages', storage_config.vm.replace('/', '_'))


def _layers_directory(config):
    return os.getcwd()


_log = logging.getLogger(__name__)
