#!/usr/bin/env python

from __future__ import print_function

import subprocess
import contextlib
import tempfile
import argparse
import shutil
import sys
import os

@contextlib.contextmanager
def tmpdir(**kwargs):
    tmp = tempfile.mkdtemp(**kwargs)
    try:
        yield tmp
    finally:
        shutil.rmtree(tmp)

# https://github.com/kata-containers/packaging/blob/master/scripts/configure-hypervisor.sh
# https://a.yandex-team.ru/arc/trunk/arcadia/infra/kernel/tools/qemu-lite/build/qemu-lite.config
# https://bb.yandex-team.ru/projects/PORTO/repos/qemu/browse/debian/rules?at=refs%2Fheads%2Fbranch_2.11.0

QEMU_CONFIG = [
        '--static',
        '--prefix=',
        '--extra-cflags=-O3 -fno-semantic-interposition -falign-functions=32 -D_FORTIFY_SOURCE=2 -fPIE',
        '--extra-ldflags=-z noexecstack -z relro -z now',
        '--target-list=x86_64-softmmu i386-softmmu',
        '--audio-drv-list=',
        '--enable-docs',       # manpages et al
        '--enable-kvm',        # -accel kvm
        '--enable-tcg',        # to work without kvm
        # ' --enable-slirp',   # -net user -- no option, enabled by default
        '--enable-tools',      # qemu-img et al
        '--enable-numa',
        '--enable-vhost-net',
        '--enable-linux-aio',  # aio=native
        '--enable-vvfat',      # -drive file=fat:/path
        '--enable-virtfs',     # -virtfs
        '--enable-attr',       # xattr
        '--enable-cap-ng',
        '--enable-curses',
        '--enable-vnc',
        '--enable-vnc-jpeg',
        '--enable-vnc-png',
        '--disable-vnc-sasl',
        '--disable-seccomp',
        '--disable-bluez',
        '--disable-brlapi',
        '--disable-gtk',        # gui
        '--disable-opengl',
        '--disable-sdl',        # gui
        '--disable-spice',
        '--disable-vte',
        '--disable-fdt',
        '--disable-glusterfs',
        '--disable-libiscsi',
        '--disable-libnfs',
        '--disable-libssh2',    # ssh block device
        '--disable-bzip2',
        '--disable-lzo',
        '--disable-snappy',
        '--disable-seccomp',
        '--disable-tpm',
        '--disable-libusb',     # usb passthrough
        '--disable-usb-redir',
        '--disable-debug-tcg',
        '--disable-tcg-interpreter',
        '--disable-qom-cast-debug',
        '--disable-tcmalloc',
        '--disable-curl',
        '--disable-rdma',
        '--disable-xen'
]

QEMU_DEPS=[
        'autoconf',
        'automake',
        'bc',
        'bison',
        'cpio',
        'flex',
        'gawk',
        'git',
        'libaudit-dev',
        'libcap-dev',
        'libcap-ng-dev',
        'libncurses5-dev',
        'libncursesw5-dev',
        'libdw-dev',
        'libelf-dev',
        'libglib2.0-0',
        'libglib2.0-dev',
        'libltdl-dev',
        'libpixman-1-dev',
        'libaio-dev',
        'libnuma-dev',
        'libtool',
        'pkg-config',
        'pkg-config',
        'python',
        'python-dev',
        'zlib1g-dev',
        'libjpeg-dev',
        'libpng-dev',
        'texinfo',         # docs
        'podlators-perl',  # docs
        'python-sphinx'    # docs
]

def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'")
    return subprocess.check_call(args, **kwargs)

def install_deps(args):
    cmd = ['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + QEMU_DEPS
    if args.dry_run:
        print("# Install qemu build dependencies.")
        print(" ".join(cmd))
    else:
        run(['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + QEMU_DEPS)

def build_qemu(args):
    with tmpdir(prefix='build-' + os.path.basename(args.src)) as build_dir:
        if os.path.isdir(args.src):
            src_dir = os.path.abspath(args.src)
        else:
            run(['tar', '--strip-components=1', '-xf', os.path.abspath(args.src)], cwd=build_dir)
            src_dir = build_dir

        help = subprocess.check_output([src_dir + '/configure', '--help'], cwd=build_dir)

        # commit b10d49d7619e4957b4b971f816661b57e5061d71
        if 'libssh2' not in help:
            QEMU_CONFIG.remove('--disable-libssh2')
            QEMU_CONFIG.append('--disable-libssh')

        run([src_dir + '/configure'] + QEMU_CONFIG, cwd=build_dir)
        run(['make', '-j', str(os.sysconf('SC_NPROCESSORS_ONLN'))], cwd=build_dir)

        if os.path.isdir(args.out):
            run(['make', 'install', 'DESTDIR=' + os.path.abspath(args.out) + '/qemu'], cwd=build_dir)
        else:
            with tmpdir(prefix='out-' + os.path.basename(args.src)) as out_dir:
                run(['make', 'install', 'DESTDIR=' + os.path.abspath(out_dir) + '/qemu'], cwd=build_dir)
                run(['tar', '-caf', os.path.abspath(args.out), 'qemu'], cwd=out_dir)

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')

    deps = subparsers.add_parser(name="install-deps")
    deps.set_defaults(handle=install_deps)
    deps.add_argument("--dry-run", default=False, action='store_true')

    build = subparsers.add_parser(name="build")
    build.set_defaults(handle=build_qemu)
    build.add_argument("src", help="directory or tarball")
    build.add_argument("out", help="directory or tarball")

    args = parser.parse_args()
    args.handle(args)
