#!/usr/bin/python

import glob
import json
import os
import sys

import pwd
import requests
import shutil
import subprocess
import tarfile
import tempfile
import xml.etree.ElementTree

try:
    import urllib3
    urllib3.disable_warnings()
except:
    pass

FNULL = open(os.devnull, 'w')

base_image_prefix = 'registry.yandex.net/platform-base-images'
sandbox_resources = {
    'porto-trusty':         369396105,
    'porto-trusty-skynet':  369403552,
    'porto-precise':        369395202,
    'porto-precise-skynet': 478916576,
    'porto-xenial':         408318239,
    'porto-xenial-skynet':  467243152,
}
images = dict([(i, None) for i in sandbox_resources.keys()])
oauth_token = os.getenv('OAUTH')
if not oauth_token:
    raise ValueError('Please setup env variable "OAUTH" with valid oauth token for access https://registry.yandex.net')
sandbox = os.path.exists('/place/sandbox-data/tasks')
build_parameters = {}

def parse_parent_from_dockerfile(source_dir):
    with open(os.path.join(source_dir, 'Dockerfile'), 'r') as f:
        for i in f.readlines():
            if i.startswith('FROM '):
                return i.split('FROM ')[-1].replace("\n", "")

def get_image_name(name):
    out = subprocess.check_output(['svn', 'log', '--xml', name])
    e = xml.etree.ElementTree.XML(out).find('logentry')
    rev, date = e.get('revision'), e.findtext('date')
    date = date.split('T')[0].replace('-', '')
    return '{}/{}:{}-{}'.format(base_image_prefix, name, date, rev)

def get_image_name_offline(name):
    out = subprocess.check_output(['svn', 'info', '--xml', name])
    e = xml.etree.ElementTree.XML(out).find('entry/commit')
    rev, date = e.get('revision'), e.findtext('date')
    date = date.split('T')[0].replace('-', '')
    return '{}/{}:{}-{}'.format(base_image_prefix, name, date, rev)

def docker_has_image(name, local=True):
    image, tag = name.split(':')
    repo_name = image.replace('registry.yandex.net/', '')
    real_registry_url = 'https://registry.yandex.net/v2/{}/manifests/{}'.format(repo_name, tag)
    registry_url = 'https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={}&tag={}'.format(image, tag)

    sys.stdout.write('Check image in remote registry ({}): '.format(repo_name+':'+tag) )
    resp = requests.get(real_registry_url, headers=dict(Authorization='OAuth {}'.format(oauth_token)), verify=False)

    if resp.status_code == 200:
        print '[present] [{}]'.format(resp.status_code)
        return True
    else:
        print '[absent] [{}]'.format(resp.status_code)

    if local:
        sys.stdout.write('Check image in local storage ({}): '.format(repo_name))
        try:
            subprocess.check_call(['docker', 'inspect', '-f', '{{ .Id }}', name], stdout=FNULL, stderr=FNULL)
            print '[present]'
            return True
        except:
            print '[absent]'
            pass

    return False

def mark_tree_rebuild(root, rebuild):
    for i in images:
        if images[i] == root:
            if rebuild:
                build_parameters[i]['rebuild'] = rebuild
            mark_tree_rebuild(i, build_parameters[i]['rebuild'])

def walk_tree(fx, filter_f, root=None):
    for i in images:
        if images[i] == root:
            ctx = build_parameters[i]
            if filter_f(ctx):
                fx(i, ctx)
            walk_tree(fx, filter_f, root=i)

def build_docker_image(source_dir, image_url):
    print 'Building {} from dir "{}"'.format(image_url, source_dir)
    print '  Patching dockerfile'
    with open(os.path.join(source_dir, 'Dockerfile')) as src:
        with open(os.path.join(source_dir, 'Dockerfile.tmp'), 'w') as dst:
            for line in src.readlines():
                if line.startswith('FROM'):
                    dst.write("FROM {}\n".format(build_parameters[images[source_dir]]['image_url']))
                else:
                    dst.write(line)

    if os.path.exists(os.path.join(source_dir, 'provision.sh')):
        print '  Preparing provision.tgz'
        with tarfile.open(os.path.join(source_dir, 'provision.tgz'), 'w:gz') as tf:
            tf.add('provision_base.sh', arcname='provision_base.sh', recursive=False)
            tf.add(os.path.join(source_dir, 'provision.sh'), arcname='provision.sh', recursive=False)

    print '  Building {}'.format(image_url)
    try:
        subprocess.check_output(
            ['docker', 'build', '--network=host', '--rm', '-f', 'Dockerfile.tmp', '-t', image_url, '.'], cwd=source_dir,
            stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError, e:
        print 'Exception while build stage\n'
        print '----------------------------\n'
        print 'return code: ', e.returncode
        print 'orig command: ', e.cmd
        print 'output (from new line):\n', e.output
        print
        raise e
    build_parameters[source_dir]['rebuild_successful'] = True


def import_porto_layer(image_url, sandbox_resource):
    print 'Importing {} from sbr:{}'.format(image_url, sandbox_resource)
    tmpdir = None
    if sandbox:
        tmpdir = os.getcwd()
    tmp = tempfile.NamedTemporaryFile(dir=tmpdir, delete=False, prefix='original_tar_')
    tmp2 = tempfile.NamedTemporaryFile(dir=tmpdir, delete=False, prefix='repacked_tar_')
    print '  Downloading sbr:{} to {}'.format(sandbox_resource, tmp.name)
    resp = requests.get('https://proxy.sandbox.yandex-team.ru/{}'.format(sandbox_resource), verify=False)
    with open(tmp.name, 'w+b') as f:
        for chunk in resp.iter_content(chunk_size=4096):
            if chunk:
                f.write(chunk)
    print '  Processing {} with tar'.format(tmp.name)
    repack_dir = tempfile.mkdtemp(prefix='unpacked_porto_layer_', dir=tmpdir)
    try:
        subprocess.check_call(['/usr/bin/sudo', 'tar', '-xf', tmp.name, '-C', repack_dir])
    except subprocess.CalledProcessError, e:
        print 'Exception while unpack : \n', e
    try:
        subprocess.check_call(['/usr/bin/sudo', 'tar', '-czf', tmp2.name, '-C', repack_dir, '.'])
    except subprocess.CalledProcessError, e:
        print 'Exception while pack: \n', e

    print '  Importing {} into local docker repo as {}'.format(tmp2.name, image_url)

    print 'file 1:', tmp.name, '  exist: ', os.path.exists(tmp.name), '  size: ', os.path.getsize(tmp.name)
    print 'file 2:', tmp2.name, '  exist: ', os.path.exists(tmp2.name), '  size: ', os.path.getsize(tmp2.name)
    print 'dir:', repack_dir, '  exist: ', os.path.exists(repack_dir), '  size: ', os.path.getsize(repack_dir)
    try:
        subprocess.check_output(['sudo', '-E', os.popen('which docker').read().strip(), 'import', tmp2.name, image_url],
                                stderr=subprocess.STDOUT,
                                env=dict({k: v for k, v in dict(os.environ).iteritems() if 'DOCKER' in k}))
    except subprocess.CalledProcessError, e:
        print 'Exception while import stage\n'
        print '----------------------------\n'
        print 'return code: ', e.returncode
        print 'orig command: ', e.cmd
        print 'output (from new line):\n', e.output

    print '  Cleanup after import porto layer'
    subprocess.check_output(["/usr/bin/sudo", "chown", '-R', pwd.getpwuid(os.getuid())[0], repack_dir])
    shutil.rmtree(repack_dir)
    os.remove(tmp.name)
    os.remove(tmp2.name)

def push_docker_image(image_url):
    print 'Pushing {}'.format(image_url)
    latest_url = image_url.split(':')[0] + ':latest'
    subprocess.check_call(['docker', 'tag', image_url, latest_url])
    subprocess.check_call(['docker', 'push', image_url])
    subprocess.check_call(['docker', 'push', latest_url])

def childrens_for(parent=None, flat_tree={}):
    return list([x for x in flat_tree.keys() if flat_tree[x] == parent])

def get_tree(node=None, flat_tree={}):
    d = {}
    d['_name'] = node
    childrens = childrens_for(node, flat_tree)
    if childrens:
        d['children'] = [get_tree(c, flat_tree) for c in childrens]
    return d

def print_tree(node=None, flat_tree={}, offset=1):
    print_name = str(node)
    if print_name == 'None':
        print_name = 'ROOT'
    print ' ' * offset, print_name
    childrens = childrens_for(node, flat_tree)
    if childrens:
        [print_tree(c, flat_tree, offset + 4) for c in childrens]


if __name__ == '__main__':
    for i in glob.glob('*/Dockerfile'):
        src_dir = os.path.dirname(i)
        parent = parse_parent_from_dockerfile(src_dir)
        if parent.startswith(base_image_prefix) and parent.endswith(':latest'):
            parent = parent.split('/')[2].split(':')[0]
        images[src_dir] = parent

    for src_dir in images.keys():
        ctx = dict(rebuild=False)
        if src_dir.startswith('porto-'):
            ctx['type'] = 'PORTO_IMPORT'
            ctx['sandbox_resource'] = sandbox_resources[src_dir]
            ctx['image_url'] = '{}/{}:{}'.format(base_image_prefix, src_dir, ctx['sandbox_resource'])
        else:
            ctx['type'] = 'DOCKER'
            ctx['image_url'] = get_image_name_offline(src_dir)
        if not docker_has_image(ctx['image_url']):
            ctx['rebuild'] = True
        build_parameters[src_dir] = ctx

    mark_tree_rebuild(None, False)

    print 'Tree of images:'
    print print_tree(None, images)
    print
    print 'Build parameters tree:'
    print json.dumps(build_parameters, sort_keys=False, indent=4)

    def fx_build(name, ctx):
        if ctx['type'] == 'DOCKER':
            build_docker_image(name, ctx['image_url'])
        elif ctx['type'] == 'PORTO_IMPORT':
            import_porto_layer(ctx['image_url'], ctx['sandbox_resource'])

    def fx_push(name, ctx):
        push_docker_image(ctx['image_url'])

    walk_tree(
        fx_build,
        lambda x: x['rebuild'])

    walk_tree(
        fx_push,
        lambda x: (docker_has_image(x['image_url'], local=True) and not docker_has_image(x['image_url'], local=False))
                  or (x['rebuild'] and x['rebuild_successful']))
