package ru.yandex.qloud.kikimr.healthcheck;

import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qloud.kikimr.jobs.ACLJob;
import ru.yandex.qloud.kikimr.transport.KikimrScheme;
import ru.yandex.qloud.kikimr.transport.acl.EAccessRights;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import java.util.Date;
import java.util.List;
import java.util.Map;

import static com.google.common.base.MoreObjects.firstNonNull;

/**
 * @author violin
 */
@Component
public class ACLUpdateHealthCheck {

    private static final String ROOT_ROBOT = "robot-qloud-kikimr";
    private static final int MIN_ROOT_USERS = 1;

    private static final int MAX_FAILED_ATTEMPTS_BEFORE_CRITICAL = 3;

    @Inject
    private HealthCheckRegistry healthCheckRegistry;

    @Inject
    private KikimrScheme kikimrScheme;

    @Inject
    private ACLJob aclJob;

    @Value("${kikimr.start.jobs}")
    private boolean startJobs;

    private Date lastCheckDate;

    @PostConstruct
    public void init() {
        if (! startJobs) {
            return;
        }

        healthCheckRegistry.register("ACL_enabled", new HealthCheck() {
            @Override
            protected Result check() throws Exception {
                return aclJob.isAclUpdateEnabled() ? Result.healthy("OK; ACL update enabled") :
                        Result.unhealthy("ATTENTION; ACL update disabled");
            }
        });

        healthCheckRegistry.register("ACL_root_users", new HealthCheck() {
            @Override
            protected HealthCheck.Result check() throws Exception {
                List<String> rootUsersWithFullAccess = kikimrScheme.getUserLoginsWithAccess("Root/qloud", EAccessRights.GENERIC_FULL);
                if (rootUsersWithFullAccess.size() < MIN_ROOT_USERS) {
                    return Result.unhealthy(
                            "too few root users with full access: %d, expected no less than %d; current root users: %s",
                            rootUsersWithFullAccess.size(), MIN_ROOT_USERS, rootUsersWithFullAccess
                    );
                }
                if (! rootUsersWithFullAccess.contains(ROOT_ROBOT)) {
                    return Result.unhealthy(
                            "%s not included in root users with full access; current root users: %s",
                            ROOT_ROBOT, rootUsersWithFullAccess
                    );
                }
                return Result.healthy("OK; current root users with full access: %s", rootUsersWithFullAccess);
            }
        });

        healthCheckRegistry.register("ACL_update_last_success", new HealthCheck() {
            @Override
            protected HealthCheck.Result check() throws Exception {
                final Date updateDate = aclJob.getLastSuccessUpdateDate();
                if (updateDate == null && lastCheckDate == null) {
                    lastCheckDate = new Date();
                    return Result.healthy("no data");
                }

                if (updateDate != null) {
                    lastCheckDate = updateDate;
                }

                final int threshold = aclJob.getJobSchedulePeriodMinutes() * 4;
                boolean healthy = DateUtils.addMinutes(lastCheckDate, threshold).after(new Date());

                final String lastSuccessMessage = updateDate != null ? "last successful ACL update on " + lastCheckDate
                        : "no data about last successful ACL update date";
                return healthy ? Result.healthy("OK; %s", lastSuccessMessage)
                        : Result.unhealthy("ACL update broken; %s", lastSuccessMessage);
            }
        });

        healthCheckRegistry.register("ACL_update_fail_attempts", new HealthCheck() {
            @Override
            protected HealthCheck.Result check() throws Exception {
                Map<String, Integer> nodeToFailedAttempts = aclJob.getKikimrTableToFailedUpdateAttempts();

                Map<String, Integer> nodeToCriticalFailedAttempts = Maps.filterEntries(
                        nodeToFailedAttempts,
                        entry -> entry != null && firstNonNull(entry.getValue(), 0) > MAX_FAILED_ATTEMPTS_BEFORE_CRITICAL
                );

                return nodeToCriticalFailedAttempts.isEmpty() ?
                        Result.healthy("OK; all node with failed ACL update attempts: %s", nodeToFailedAttempts) :
                        Result.unhealthy(
                                "FAIL; %d nodes have more than %d failed ACL update attempts : %s; all node with failed ACL update attempts: %s",
                                nodeToCriticalFailedAttempts.size(), MAX_FAILED_ATTEMPTS_BEFORE_CRITICAL,
                                nodeToCriticalFailedAttempts, nodeToFailedAttempts
                        );
            }
        });

    }
}
