package ru.yandex.qloud.kikimr.jobs;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.Uninterruptibles;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.auth.api.Group;
import ru.yandex.qe.auth.api.User;
import ru.yandex.qloud.kikimr.jobs.qloud.MergedAuthApiClientService;
import ru.yandex.qloud.kikimr.transport.KikimrScheme;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.MoreObjects.firstNonNull;

/**
 * @author violin
 */
@Component
public class ACLJob {
    private final static Logger LOG = LoggerFactory.getLogger(ACLJob.class);

    private static final int JOB_SCHEDULE_PERIOD_MINUTES = 20;

    private static final Set<String> GLOBAL_ROLES = Sets.newHashSet("qloud-production", "qloud-external");
    private static final Set<String> TITLES = Sets.newHashSet("ADMINISTRATOR", "USER", "MAINTAINER", "OWNER", "ROOT_SSH");


    private final ScheduledExecutorService scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();

    private final Multimap<String, String> kikimrNodeToLoginsCache = MultimapBuilder.hashKeys().hashSetValues().build();
    
    @Inject
    private KikimrScheme kikimrScheme;

    @Inject
    private MergedAuthApiClientService authService;

    @Value("${kikimr.start.jobs}")
    private boolean startJobs;
    @Value("${env.KIKIMR_ACL_UPDATE_ENABLED:false}")
    private boolean aclUpdateEnabled;
    

    private volatile Date lastSuccessUpdateDate = null;

    private Map<String, Integer> kikimrTableToFailedUpdateAttempts = Maps.newConcurrentMap();

    @PostConstruct
    private void init() {
        if (startJobs) {
            scheduledExecutorService.scheduleAtFixedRate(() -> {
                try {
                    LOG.info("starting update acl");
                    updateAcl();
                    lastSuccessUpdateDate = new Date();
                    LOG.info("acl update finished");
                } catch (Exception e) {
                    LOG.error("acl update failed", e);
                }
            }, 0L, JOB_SCHEDULE_PERIOD_MINUTES, TimeUnit.MINUTES);
        } else {
            LOG.info("jobs disabled; acl update not started");
        }
    }

    public boolean isAclUpdateEnabled() {
        return aclUpdateEnabled;
    }

    public Date getLastSuccessUpdateDate() {
        return lastSuccessUpdateDate;
    }

    public Map<String, Integer> getKikimrTableToFailedUpdateAttempts() {
        return ImmutableMap.copyOf(kikimrTableToFailedUpdateAttempts);
    }

    public int getJobSchedulePeriodMinutes() {
        return JOB_SCHEDULE_PERIOD_MINUTES;
    }

    private void updateAcl() {
        if (! aclUpdateEnabled) {
            LOG.info("acl update disabled");
            return;
        }

        LOG.info("update acl started");
        Multimap<String, String> nodeToUsers = MultimapBuilder.hashKeys().hashSetValues().build();
        Set<String> globalAdmins = Sets.newHashSetWithExpectedSize(200);
        for (User user : authService.listAllUsers()) {
            for (String role : Iterables.concat(user.getRoles(), user.getAdminRoles())) {
                if (isGlobal(role)) {
                    globalAdmins.add(user.getLogin());
                } else {
                    nodeToUsers.put(toNode(role), user.getLogin());              
                }
            }
        }
        for (Group group : authService.listAllGroups()) {
            for (String role : Iterables.concat(group.getRoles(), group.getAdminRoles())) {
                nodeToUsers.put(toNode(role), group.getUrl());              
            }
        }
        LOG.info("qloud: found {} roles; global admins ({}) : {}", nodeToUsers.size(), globalAdmins.size(), globalAdmins);

        Set<String> kikimrNodes = kikimrScheme.listQloudNodesWithoutTables();
        LOG.info("kikimr: {} nodes to update ACL", nodeToUsers.size());

        long startTime = System.currentTimeMillis();
        boolean cacheInitialized = kikimrNodeToLoginsCache.size() > 0;
        
        for (String kikimrNode : kikimrNodes) {
            Collection<String> users = nodeToUsers.get(kikimrNode);
            if ((! (cacheInitialized && users.equals(kikimrNodeToLoginsCache.get(kikimrNode)))) || kikimrTableToFailedUpdateAttempts.containsKey(kikimrNode)) {
                Uninterruptibles.sleepUninterruptibly(50, TimeUnit.MILLISECONDS);
                try {
                    LOG.debug("setting USER role for node {} and users {}", kikimrNode, users);
                    kikimrScheme.setUserACL(kikimrNode, users);
                    kikimrNodeToLoginsCache.putAll(kikimrNode, users);
                    kikimrTableToFailedUpdateAttempts.remove(kikimrNode);
                } catch (Exception e) {
                    kikimrTableToFailedUpdateAttempts.put(kikimrNode, firstNonNull(kikimrTableToFailedUpdateAttempts.get(kikimrNode), 0) + 1);
                    LOG.error(String.format("fail to update ACL for node %s and users %s", kikimrNode, users), e);
                }
            }
        }
        LOG.info("USER kikimr ACL updates finished in {} ms", System.currentTimeMillis() - startTime);

        LOG.debug("setting ADMIN role for node Root/qloud and users {}", globalAdmins);
        kikimrScheme.setAdminACL("Root/qloud", globalAdmins);

        LOG.info("ACL update finished");
    }

    private boolean isGlobal(String role) {
        return GLOBAL_ROLES.contains(role);
    }

    private String toNode(final String role) {
        String node = role;
        for (String global : GLOBAL_ROLES) {
            if (node.startsWith(global + ".")) {
                node = node.substring(global.length() + 1);
            }
        }
        for (String title : TITLES) {
            if (node.endsWith("." + title)) {
                node = node.substring(0, node.length() - title.length() - 1);
            }
        }
        return "Root/qloud/" + node.replaceAll("\\.", "/");
    }

}
