package ru.yandex.qloud.kikimr.jobs;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.Uninterruptibles;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qloud.kikimr.jobs.qloud.LoggerSettings;
import ru.yandex.qloud.kikimr.jobs.qloud.MergedQloudApiClientService;
import ru.yandex.qloud.kikimr.transport.KQPException;
import ru.yandex.qloud.kikimr.transport.KikimrScheme;
import ru.yandex.qloud.kikimr.transport.YQL;
import ru.yandex.qloud.kikimr.transport.YQLException;
import ru.yandex.qloud.kikimr.utils.TableUtils;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author violin
 */
@Component
public class DropTablesByTTLJob {
    private final static Logger LOG = LoggerFactory.getLogger(DropTablesByTTLJob.class);

    private static final int JOB_SCHEDULE_PERIOD_MINUTES = 120;

    private final ScheduledExecutorService scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();

    private final int dropBatchSize = 50;

    private volatile Date lastSuccessCleanupDate = null;

    @Inject
    private MergedQloudApiClientService qloudService;

    @Inject
    private YQL yql;
    @Inject
    private KikimrScheme kikimrScheme;

    @Value("${env.KIKIMR_MIN_DAYS_KEEP_LOG:15}")
    private int minDaysToKeepLog;
    @Value("${env.KIKIMR_DEFAULT_DAYS_KEEP_LOG:15}")
    private int defaultDaysToKeepLog;
    @Value("${env.KIKIMR_MAX_DAYS_KEEP_LOG:30}")
    private int maxDaysToKeepLog;

    @Value("${kikimr.start.jobs}")
    private boolean startJobs;
    @Value("${env.KIKIMR_TABLES_TTL_CLEANUP_ENABLED:false}")
    private boolean cleanupEnabled;

    @PostConstruct
    private void init() {
        if (startJobs) {
            scheduledExecutorService.scheduleAtFixedRate(() -> {
                try {
                    LOG.info("starting cleanup; previous successful cleanup date = {}", lastSuccessCleanupDate);
                    dropTablesByTTL();
                    lastSuccessCleanupDate = new Date();
                    LOG.info("cleanup finished");
                } catch (Exception e) {
                    LOG.error("Cleanup failed", e);
                }
            }, 0L, JOB_SCHEDULE_PERIOD_MINUTES, TimeUnit.MINUTES);
        } else {
            LOG.info("jobs disabled; table cleanup not started");
        }
    }

    public Date getLastSuccessCleanupDate() {
        return lastSuccessCleanupDate;
    }

    public int getMaxDaysToKeepLog() {
        return maxDaysToKeepLog;
    }

    public int getJobSchedulePeriodMinutes() {
        return JOB_SCHEDULE_PERIOD_MINUTES;
    }

    public boolean isCleanupEnabled() {
        return cleanupEnabled;
    }

    public void dropTablesByTTL() {
        if (! cleanupEnabled) {
            LOG.info("cleanup disabled, expired tables won't be dropped");
            return;
        }

        final List<LoggerSettings> loggerSettingsList = qloudService.getLoggerSettings();
        LOG.info("got settings from qloud, count = {}", loggerSettingsList.size());
        final Set<String> kikimrTables = kikimrScheme.listAllQloudTables();
        LOG.info("got tables from kikimr, count = {}", kikimrTables.size());

        final Map<String, Integer> environmentToTTL = buildTTLMap(loggerSettingsList);
        final List<String> tablesToDrop = getTablesToDrop(kikimrTables, environmentToTTL);
        LOG.info("got {} tables to drop : {}", tablesToDrop.size(), tablesToDrop);

        dropTables(tablesToDrop);
        LOG.info("drop expired tables done");
    }

    private void dropTables(List<String> tables) {
        Iterables.partition(tables, dropBatchSize).forEach(
                (partTables) -> {
                    String dropQuery = StringUtils.join(
                            partTables.stream().map((table) -> String.format("DROP TABLE [%s]", table)).collect(Collectors.toList()),
                            "; "
                    ) + ";";
                    try {
                        yql.executeQueryWithRootAccess(dropQuery);
                    } catch (YQLException|KQPException e) {
                        LOG.warn("tables " + partTables + " not dropped", e);
                    }
                    kikimrScheme.invalidateSchemeCache();
                    Uninterruptibles.sleepUninterruptibly(100, TimeUnit.MILLISECONDS);
                }
        );
    }

    private Map<String, Integer> buildTTLMap(final List<LoggerSettings> loggerSettings) {
        final Map<String, Integer> environmentToTTL = Maps.newHashMapWithExpectedSize(loggerSettings.size());
        loggerSettings.forEach(
                (settings) -> {
                    final String environment = settings.getEnvironmentLocator().getObjectId();
                    final int ttl = environmentToTTL.containsKey(environment) ?
                            Math.max(environmentToTTL.get(environment), settings.getEsIndexTtl()) :
                            settings.getEsIndexTtl();
                    environmentToTTL.put(environment, Math.min(Math.max(ttl, minDaysToKeepLog), maxDaysToKeepLog));
                }
        );
        return environmentToTTL;
    }

    private List<String>  getTablesToDrop(final Collection<String> kikimrTables, final Map<String, Integer> environmentToTTL) {
        final LocalDate currentDate = LocalDate.now();
        final ImmutableList.Builder<String> tablesToDrop = ImmutableList.builder();
        kikimrTables.forEach(
                (table) -> {
                    Integer ttl = environmentToTTL.get(getEnvironment(table));
                    if (ttl == null) {
                        ttl = defaultDaysToKeepLog;
                        LOG.debug("TTL is null for {} ({}), setting to default ({})", getEnvironment(table), table, ttl);
                    }

                    LocalDate tableDate = TableUtils.parseDateFromTableName(table);
                    if (tableDate != null && tableDate.isBefore(currentDate.minusDays(ttl))) {
                        tablesToDrop.add(table);
                    }
                }
        );
        return tablesToDrop.build();
    }

    private String getEnvironment(String kikimrTable) {
        String withoutPrefix = StringUtils.substringAfter(kikimrTable, "Root/qloud/");
        return withoutPrefix.substring(0, withoutPrefix.lastIndexOf("/")).replaceAll("/", ".");
    }
}
