package ru.yandex.qloud.kikimr.jobs;

import com.codahale.metrics.ExponentiallyDecayingReservoir;
import com.codahale.metrics.Histogram;
import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.telemetry.metrics.Gauges;
import ru.yandex.qe.telemetry.metrics.yasm.YasmMetricRegistry;
import ru.yandex.qloud.kikimr.scheme.TableStatistics;
import ru.yandex.qloud.kikimr.transport.KikimrScheme;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author violin
 */
@Component
public class KikimrTableStatisticsJob {
    private final static Logger LOG = LoggerFactory.getLogger(KikimrTableStatisticsJob.class);

    private static final Set<String> DEFAULT_GAUGE_AGGREGATION = Sets.newHashSet("axxt");

    private final ScheduledExecutorService scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();

    @Value("${kikimr.start.jobs}")
    private boolean startJobs;

    @Value("${env.KIKIMR_TABLE_STATISTICS_ENABLED:false}")
    private boolean kikimrTableStatisticsEnabled;

    @Inject
    private YasmMetricRegistry metricRegistry;

    @Inject
    private KikimrScheme kikimrScheme;
    
    @PostConstruct
    private void init() {
        if (startJobs) {
            scheduledExecutorService.scheduleAtFixedRate(() -> {
                try {
                    LOG.info("starting updating table statistics");
                    updateKikimrTableMetrics();
                    LOG.info("table statistics update finished");
                } catch (Exception e) {
                    LOG.error("table statistics update failed", e);
                }
            }, 0L, 30, TimeUnit.MINUTES);
        } else {
            LOG.info("jobs disabled; acl update not started");
        }
    }

    private void updateKikimrTableMetrics() {
        if (! kikimrTableStatisticsEnabled) {
            LOG.info("table statistics update disabled");
            return;
        }

        Set<String> allNodes;
        Set<String> tables;
        do {
            LOG.debug("loading nodes from scheme...");
            allNodes = kikimrScheme.listAllQloudNodesAndTables();
            tables = kikimrScheme.listAllQloudTables();
        } while (! allNodes.containsAll(tables));

        int excludedTablesFromStatCount = 0;

        long sumRows = 0;
        long sumSize = 0;
        LOG.debug("getting table statistics...");

        Histogram rowsHistogram = new Histogram(new ExponentiallyDecayingReservoir());
        Histogram sizeHistogram = new Histogram(new ExponentiallyDecayingReservoir());
        for (String table : tables) {
            try {
                TableStatistics tableStatistics = kikimrScheme.getTableStatistics(table);

                sumRows += tableStatistics.getRowCount();
                sumSize += tableStatistics.getDataSize();

                rowsHistogram.update(tableStatistics.getRowCount());
                sizeHistogram.update(tableStatistics.getDataSize());

            } catch (Exception e) {
                LOG.warn(String.format("table %s excluded from statistics", table), e);
                excludedTablesFromStatCount++;
            }
        }

        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "rows", "p50"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(rowsHistogram.getSnapshot().getValue(0.5));
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "rows", "p75"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(rowsHistogram.getSnapshot().get75thPercentile());
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "rows", "p95"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(rowsHistogram.getSnapshot().get95thPercentile());
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "rows", "p99"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(rowsHistogram.getSnapshot().get99thPercentile());

        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "size", "p50"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(sizeHistogram.getSnapshot().getValue(0.5));
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "size", "p75"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(sizeHistogram.getSnapshot().get75thPercentile());
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "size", "p95"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(sizeHistogram.getSnapshot().get95thPercentile());
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "tables", "size", "p99"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(sizeHistogram.getSnapshot().get99thPercentile());

        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "summary", "tables", "rows", "sum"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(sumRows);
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "summary", "tables", "size", "sum"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(sumSize);

        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "summary", "tables", "count"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(tables.size());
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "summary", "nodes", "count"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(allNodes.size());
        Gauges.settableGauge(
                metricRegistry,
                MetricRegistry.name(KikimrTableStatisticsJob.class, "kikimr", "summary", "tables", "excluded", "count"),
                DEFAULT_GAUGE_AGGREGATION
        ).setValue(excludedTablesFromStatCount);
    }
}
