package ru.yandex.qloud.kikimr.lucene;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import org.springframework.stereotype.Component;
import ru.yandex.qloud.kikimr.transport.YQL;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import static java.util.stream.Collectors.toList;

@Component
public class ESResultConverter {
    private final static ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    public ESResult convert(YQL.Result yqlResult) {
        List<String> fields = yqlResult.getFieldsNames();

        return new ESResult(yqlResult.getRows().stream()
                .map(r -> {
                    Object[] values = r.getData();
                    Map<String, Object> data = Maps.newHashMap();
                    for (int i = 0; i < fields.size(); i++) {
                        data.put(fields.get(i), values[i]);
                    }
                    return new Hit(data);
                })
                .collect(toList()));
    }

    public static class ESResult {
        private final Hits hits;

        ESResult(List<Hit> hits) {
            this.hits = new Hits(hits);
        }

        public Hits getHits() {
            return hits;
        }
    }

    public static class Hits {
        private final List<Hit> hits;

        Hits(List<Hit> hits) {
            this.hits = hits;
        }

        @JsonProperty("_hits")
        public List<Hit> getHits() {
            return hits;
        }
    }

    public static class Hit {
        private final Map<String, Object> source;

        Hit(Map<String, Object> source) {
            this.source = source;
            normalize();
        }

        private void normalize() {
            try {
                if (source.containsKey("timestamp_raw")) {
                    source.put("@timestamp", source.get("timestamp_raw"));
                }
                if (source.containsKey("fields")) {
                    String fields = (String) source.get("fields");
                    if (fields.isEmpty()) {
                        fields = "{}";
                    }
                    JsonNode fieldsNode = OBJECT_MAPPER.readTree(fields);
                    source.put("@fields", fieldsNode);
                    source.remove("fields");
                }
                if (source.containsKey("rest")) {
                    String rest = (String) source.get("rest");
                    if (rest.isEmpty()) {
                        rest = "{}";
                    }
                    JsonNode restNode = OBJECT_MAPPER.readTree(rest);
                    source.put("@rest", restNode);
                    source.remove("rest");
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        @JsonProperty("_source")
        public Map<String, Object> getSource() {
            return source;
        }
    }
}
