package ru.yandex.qloud.kikimr.search;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Component;
import ru.yandex.qloud.kikimr.lucene.ESQueryConverter;
import ru.yandex.qloud.kikimr.lucene.PagingParameters;
import ru.yandex.qloud.kikimr.lucene.TimeRangeFilter;
import ru.yandex.qloud.kikimr.transport.KikimrScheme;
import ru.yandex.qloud.kikimr.utils.TableUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.function.BiFunction;

import static ru.yandex.qloud.kikimr.utils.TableUtils.toKikimrTable;

@Component
public class KikimrQueryRequestFactory {

    private static final DateTimeFormatter TABLE_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final String FAKE_TABLE = "##KIKIMR_TABLE##";

    @Inject
    private KikimrScheme kikimrScheme;

    @Inject
    private ESQueryConverter esQueryConverter;

    public KikimrQueryRequest createKikimrQueryRequest(
            @Nonnull QueryWhereCondition queryWhereCondition,
            @Nonnull String kikimrEnvPath,
            @Nullable PagingParameters pagingParameters,
            @Nonnull TimeRangeFilter timeRangeFilter,
            @Nullable String defaultDate,
            boolean isAccessLog
    ) {
        List<String> tableDates = getTableDates(kikimrEnvPath, pagingParameters, timeRangeFilter, defaultDate);

        String parameterizedQuery = KikimrYqlQueryBuilder.buildYqlQuery(FAKE_TABLE, queryWhereCondition, pagingParameters, timeRangeFilter, isAccessLog);

        ImmutableList.Builder<String> tablesBuilder = ImmutableList.builder();
        ImmutableList.Builder<String> kikimrQueriesBuilder = ImmutableList.builder();

        tableDates.forEach(
                date -> {
                    String table = toKikimrTable(kikimrEnvPath, date, isAccessLog);
                    tablesBuilder.add(table);
                    kikimrQueriesBuilder.add(StringUtils.replace(parameterizedQuery, FAKE_TABLE, table));
                }
        );

        return new KikimrQueryRequest(queryWhereCondition, pagingParameters, timeRangeFilter, tablesBuilder.build(), kikimrQueriesBuilder.build());
    }

    public KikimrQueryRequest createEmptyQueryRequest(
            @Nonnull QueryWhereCondition queryWhereCondition,
            @Nullable PagingParameters pagingParameters,
            @Nonnull TimeRangeFilter timeRangeFilter
    ) {
        return new KikimrQueryRequest(queryWhereCondition, pagingParameters, timeRangeFilter, Collections.emptyList(), Collections.emptyList());
    }

    private List<String> getTableDates(
            @Nonnull String kikimrEnvPath,
            @Nullable PagingParameters pagingParameters,
            @Nonnull TimeRangeFilter timeRangeFilter,
            @Nullable String defaultDate
    ) {
        PagingParameters.PagingDirection pagingDirection = getPagingDirection(pagingParameters);

        MinMaxTimestamps minMaxTimestamps = getMinMaxTimestamps(kikimrEnvPath, defaultDate, pagingParameters, timeRangeFilter);
        if (minMaxTimestamps == null) {
            return defaultDate != null ? Collections.singletonList(defaultDate) : Collections.singletonList(LocalDateTime.now().format(DateTimeFormatter.ISO_DATE));
        }

        List<String> result = Lists.newLinkedList();
        ZonedDateTime currentDateTime =  ZonedDateTime.ofInstant(Instant.ofEpochMilli(minMaxTimestamps.getMinTimestamp()), ZoneId.of("Z"));
        ZonedDateTime maxDateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(minMaxTimestamps.getMaxTimestamp()), ZoneId.of("Z"));
        boolean shouldContinue = maxDateTime.isAfter(currentDateTime);
        while (shouldContinue) {
            result.add(currentDateTime.format(TABLE_DATE_FORMATTER));
            ZonedDateTime nextDateTime = currentDateTime.plusDays(1);
            shouldContinue = ! (nextDateTime.isAfter(maxDateTime) && currentDateTime.getDayOfYear() == maxDateTime.getDayOfYear());
            currentDateTime = nextDateTime;
        }

        if (pagingDirection == PagingParameters.PagingDirection.PREV) {
            Collections.reverse(result);
        }

        return result;
    }

    @Nullable
    private MinMaxTimestamps getMinMaxTimestamps(
            @Nonnull String kikimrEnvPath,
            @Nullable String defaultDate,
            @Nullable PagingParameters pagingParameters,
            @Nonnull TimeRangeFilter timeRangeFilter
    ) {
        Long minTimestamp = null;
        Long maxTimestamp = null;

        if (pagingParameters != null) {
            if (getPagingDirection(pagingParameters) == PagingParameters.PagingDirection.PREV) {
                maxTimestamp = -pagingParameters.getTimestamp();
            } else {
                minTimestamp = -pagingParameters.getTimestamp();
            }
        }

        maxTimestamp = nullOrCompute(maxTimestamp, timeRangeFilter.getTimestampTo(), Math::max);
        minTimestamp = nullOrCompute(minTimestamp, timeRangeFilter.getTimestampFrom(), Math::min);

        if (maxTimestamp == null && minTimestamp == null && defaultDate != null) {
            return null;
        }

        if (maxTimestamp == null || minTimestamp == null) {
            List<String> availableTables = kikimrScheme.getAvailableTablesSimpleNames(kikimrEnvPath);
            if (availableTables.isEmpty()) {
                return null;
            }

            if (maxTimestamp == null) {
                maxTimestamp = ZonedDateTime.parse(TableUtils.unparsedDateFromTableName(availableTables.get(availableTables.size() - 1)) + "T23:59:59Z").toInstant().toEpochMilli();
            }
            if (minTimestamp == null) {
                minTimestamp = ZonedDateTime.parse(TableUtils.unparsedDateFromTableName(availableTables.get(0)) + "T00:00:00Z").toInstant().toEpochMilli();
            }
        }

        return new MinMaxTimestamps(minTimestamp, maxTimestamp);
    }

    @Nonnull
    private PagingParameters.PagingDirection getPagingDirection(@Nullable PagingParameters pagingParameters) {
        return (pagingParameters != null) ? pagingParameters.getPagingDirection() :
                PagingParameters.PagingDirection.PREV;
    }

    private Long nullOrCompute(Long value1, Long value2, BiFunction<Long, Long, Long> function) {
        if (value1 == null && value2 == null) {
            return null;
        }
        if (value1 == null) {
            return value2;
        }
        if (value2 == null) {
            return value1;
        }
        return function.apply(value1, value2);
    }

    private static class MinMaxTimestamps {
        private final long minTimestamp;
        private final long maxTimestamp;

        public MinMaxTimestamps(@Nonnull long minTimestamp, @Nonnull long maxTimestamp) {
            this.minTimestamp = minTimestamp;
            this.maxTimestamp = maxTimestamp;
        }

        public long getMinTimestamp() {
            return minTimestamp;
        }

        public long getMaxTimestamp() {
            return maxTimestamp;
        }
    }
}
